-- Regression tests for period

CREATE TEMP TABLE regress_period(
  id   INT,
  span PERIOD);

CREATE TEMP TABLE regress_periods(
  id    INT,
  spans PERIOD[]);

-- TRUNCATE TABLE regress_periods
INSERT INTO regress_period VALUES 
(1, period('2000-01-01'::date)),
(2, period('2000-01-01 8:00:00'::timestamptz, INTERVAL '8 hours')),
(3, period('2000-01-01 16:00:00'::timestamptz, INTERVAL '8 hours')),
(4, period('2000-01-01 12:00:00'::timestamptz, INTERVAL '4 hours')),
(5, period('2000-01-01 8:00:00'::timestamptz, INTERVAL '2 hours'));

INSERT INTO regress_periods
SELECT 1, array_agg(span)
FROM regress_period;

/***************************************************************************
*                          constructors
***************************************************************************/

SELECT period('[2000-01-01 00:00:00, 2000-01-02 00:00:00)') = span
FROM regress_period WHERE id = 1;

SELECT period('2000-01-01 00:00:00'::timestamptz, 
    '2000-01-02 00:00:00'::timestamptz) = span
FROM regress_period WHERE id = 1;

SELECT period('2000-01-01'::timestamptz, 86400) = span
FROM regress_period WHERE id = 1;

SELECT period('2000-01-01'::timestamptz, INTERVAL '1d') = span
FROM regress_period WHERE id = 1;

SELECT period('2000-01-01'::date) = span
FROM regress_period WHERE id = 1;


/***************************************************************************
*                          fact functions
***************************************************************************/

SELECT prior(span), first(span), last(span), next(span)
FROM regress_period WHERE id = 1;

SELECT length(span), seconds(span), mean_time(span)
FROM regress_period WHERE id = 1;


/***************************************************************************
*                        boolean functions
***************************************************************************/

SELECT spans[2] << spans[3] AS before,
  spans[3] >> spans[2] AS after,
  spans[2] >< spans[3] AS adjacent, 
  spans[2] && spans[4] AS overlaps
FROM regress_periods;

SELECT spans[1] @> spans[2] AS contains, 
  spans[1] @> prior(spans[2]) AS contains_ts
FROM regress_periods;

SELECT spans[2] <@ spans[1] AS contained_by, 
  prior(spans[2]) <@ spans[1] AS contained_by_ts
FROM regress_periods;

SELECT starts(spans[5],spans[2]) AS starts,
  ends(spans[3], spans[1]) AS ends
FROM regress_periods;

SELECT spans[2] + spans[4] AS period_union,
  spans[2] & spans[4] AS period_intersect
FROM regress_periods;

SELECT enumerate(span, INTERVAL '1 hour')
FROM regress_period WHERE id = 2;

SELECT span + INTERVAL '30 minutes' AS period_grow,
span - INTERVAL '30 minutes' AS period_shrink,
period_shift(span, INTERVAL '30 minutes') 
FROM regress_period WHERE id = 2;

SELECT spans[2] << spans[3] AS before,
  spans[3] >> spans[2] AS after
FROM regress_periods;

DROP TABLE regress_period;
DROP TABLE regress_periods;
