/* $Id: heaptuple.c,v 1.16 2006/02/27 04:41:44 jwp Exp $
 *
 * Copyright 2005, PostgresPy Project.
 * http://python.projects.postgresql.org
 *
 *//*
 * Postgres HeapTuple interface for Python
 *
 * This implements an interface to HeapTuples that are of a TupleDesc that does
 * not specify the columns of a Relation.
 *
 * It is the interface for HeapTuples that are not a registered composite type.
 */
#include <setjmp.h>
#include <postgres.h>
#include <tcop/tcopprot.h>
#include <access/htup.h>
#include <access/hio.h>
#include <access/heapam.h>
#include <access/tupdesc.h>
#include <catalog/pg_type.h>
#include <lib/stringinfo.h>
#include <parser/parse_type.h>
#include <utils/array.h>
#include <utils/datum.h>
#include <utils/relcache.h>
#include <utils/syscache.h>
#include <utils/typcache.h>
#include <utils/tuplestore.h>
#include <pypg/postgres.h>
#include <pypg/environment.h>

#include <Python.h>
#include <structmember.h>
#include <pypg/python.h>

#include <pypg/externs.h>
#include <pypg/error.h>
#include <pypg/tupledesc.h>
#include <pypg/heaptuple.h>
#include <pypg/type.h>
#include <pypg/type/object.h>
#include <pypg/type/system.h>

HeapTuple
PyPg_heap_copytuple(HeapTuple ht)
{
	HeapTuple nht = NULL;
	PythonMemoryContext(nht = heap_copytuple(ht));
	return(nht);
}

PyObj EmptyPyPgHeapTuple = NULL;

int
EmptyPyPgHeapTuple_Initialize(void)
{
	TupleDesc td;
	HeapTuple ht = NULL;

	td = PyPgTupleDesc_FetchTupleDesc(EmptyPyPgTupleDesc);
	PgError_TRAP(ht = heap_formtuple(td, NULL, NULL));
	if (PyErr_Occurred()) return(-1);

	EmptyPyPgHeapTuple = PyPgHeapTuple_New(EmptyPyPgTupleDesc, ht);
	if (ht != NULL) heap_freetuple(ht);
	if (EmptyPyPgHeapTuple == NULL) return(-1);
	return(0);
}

HeapTupleHeader
HeapTupleHeader_FromHeapTuple(HeapTuple ht)
{
	HeapTupleHeader rot;

	rot = (HeapTupleHeader) palloc(ht->t_len);

	memcpy(rot, ht->t_data, ht->t_len);
	HeapTupleHeaderSetDatumLength(rot, ht->t_len);
	HeapTupleHeaderSetTypMod(rot, -1);
	if (OidIsValid(ht->t_tableOid))
	{
		Relation rd = NULL;
		Oid reltype = RECORDOID;

		rd = RelationIdGetRelation(ht->t_tableOid);
		if (rd != NULL)
		{
			reltype = rd->rd_rel->reltype;
			RelationClose(rd);
		}

		HeapTupleHeaderSetTypeId(rot, reltype);
	}
	else
		HeapTupleHeaderSetTypeId(rot, RECORDOID);

	return(rot);
}

HeapTuple
HeapTuple_FromTupleDescAndDatumNulls(TupleDesc td, Datum *datums, bool *nulls)
{
	int i;
	char cnulls[FUNC_MAX_ARGS];
	HeapTuple ht;

	if (td->natts > 0)
		for (i = 0; i < td->natts; ++i)
			cnulls[i] = nulls[i] ? 'n' : ' ';
	ht = heap_formtuple(td, datums, cnulls);

	return(ht);
}

/*
 * HeapTuple_FromTupleDescAndIterable
 *
 * Convert a Python iterable to a HeapTuple
 */
HeapTuple
HeapTuple_FromTupleDescAndIterable(TupleDesc td, PyObj so)
{
	int natts = td->natts;
	PyObj attr = NULL;
	
	unsigned int i = 0;
	char *nulls = NULL;
	Datum *values = NULL;
	PyObj *obvalues = NULL;
	HeapTuple ht = NULL;
	PyObj iter;

	iter = PyObject_GetIter(so);
	if (iter == NULL) return(NULL);

	PG_TRY();
	{
		nulls = palloc(sizeof(char) * natts);
		values = palloc(sizeof(Datum) * natts);
		obvalues = palloc(sizeof(PyObj) * natts);
	}
	PG_CATCH();
	{
		PyErr_SetPgError();
		Py_DECREF(iter);

		if (nulls) pfree(nulls);
		if (values) pfree(values);
	}
	PG_END_TRY();
	if (PyErr_Occurred()) return(NULL);

	while ((attr = PyIter_Next(iter)))
	{
		/* Ignore the overflow */
		if (i >= natts) break;

		if (attr == Py_None)
		{
			nulls[i] = 'n';
			values[i] = 0;
			obvalues[i] = attr;
		}
		else
		{
			nulls[i] = ' ';

			if (td->attrs[i]->atttypid == BOOLOID)
			{
				values[i] = PyObject_IsTrue(attr) ? 1 : 0;
				Py_DECREF(attr);
				if (PyErr_Occurred()) goto loop_error;

				obvalues[i] = Py_None;
				Py_INCREF(Py_None);
			}
			else
			{
				PyObj ob, typ;
				typ = PyPgType_FromOid(td->attrs[i]->atttypid);
				if (typ == NULL)
				{
					Py_DECREF(attr);
					goto loop_error;
				}
				ob = obvalues[i] = Py_Call(typ, attr);
				Py_DECREF(attr);
				if (ob == NULL) goto loop_error;

				values[i] = PyPgObject_FetchDatum(ob);
			}
		}
		++i;
	}
	Py_DECREF(iter);

	/* Handle underflow by NULLing out the rest */
	if (i < td->natts)
		for (; i < td->natts; ++i)
		{
			nulls[i] = 'n';
			values[i] = 0;
			obvalues[i] = Py_None;
			Py_INCREF(Py_None);
		}

	PgError_TRAP(ht = heap_formtuple(td, values, nulls));

	for (i = 0; i < natts; ++i)
	{
		Py_DECREF(obvalues[i]);
	}
	pfree(obvalues);
	pfree(values);
	pfree(nulls);

	return(ht);
loop_error:
	{
		int pos = i;
		Py_DECREF(iter);
		pfree(nulls);
		pfree(values);
		for (i = 0; i < pos; ++i)
		{
			Py_DECREF(obvalues[i]);
		}
		pfree(obvalues);
	}
	return(NULL);
}

/*
 * HeapTuple_FromTupleDescAndPyObject
 *
 * Place a Python object into a new single attribute HeapTuple
 */
HeapTuple
HeapTuple_FromTupleDescAndPyObject(TupleDesc td, PyObj ob)
{
	char null;
	Datum value;
	HeapTuple ht = NULL;

	Assert(td->natts == 1);

	if (ob == Py_None)
	{
		null = 'n';
		value = 0;
		Py_INCREF(ob);
	}
	else
	{
		if (td->attrs[0]->atttypid == BOOLOID)
		{
			value = PyObject_IsTrue(ob) ? 1 : 0;
			if (PyErr_Occurred()) return(NULL);
			Py_INCREF(ob);
		}
		else
		{
			PyObj source, typ;
			typ = PyPgType_FromOid(td->attrs[0]->atttypid);
			if (typ == NULL) return(NULL);
			source = ob;
			ob = Py_Call(typ, source);
			Py_DECREF(typ);
			if (ob == NULL) return(NULL);

			Assert(PyPgObject_Check(ob));
			value = PyPgObject_FetchDatum(ob);
		}

		null = ' ';
	}

	PgError_TRAP(ht = heap_formtuple(td, &value, &null));
	Py_DECREF(ob);

	return(ht);
}

static int
tup_length(PyObj self)
{
	return(PyPgHeapTuple_FetchNatts(self));
}

static PyObj
tup_item(PyObj self, int attnum)
{
	bool isnull = false;
	TupleDesc td;
	HeapTuple ht;
	int natts;
	Datum datum = 0;
	PyObj atttyp, tdo, rob;

	ht = PyPgHeapTuple_FetchHeapTuple(self);
	natts = HeapTuple_FetchNatts(ht);

	if (attnum >= natts)
	{
		PyErr_Format(PyExc_IndexError,
				"index(%d) out of range(%d)", attnum, natts);
		return(NULL);
	}

	tdo = PyPgHeapTuple_FetchPyPgTupleDesc(self);
	atttyp = PyPgTupleDesc_FetchPyPgType(tdo, attnum);
	if (atttyp == NULL) return(NULL);

	td = PyPgTupleDesc_FetchTupleDesc(tdo);
	PgError_TRAP(datum = fastgetattr(ht, attnum + 1, td, &isnull));
	if (PyErr_Occurred()) return(NULL);

	if (isnull)
	{
		rob = Py_None;
		Py_INCREF(rob);
	}
	else
		rob = PyPgObject_FromPyPgTypeAndDatum(atttyp, datum);

	return(rob);
}

static PyObj
tup_slice(PyObj self, int from, int to)
{
	PyObj rob;
	int i;

	if (to == INT_MAX) to = PyPgHeapTuple_FetchTupleDesc(self)->natts;

	rob = PyTuple_New(to - from);
	if (rob == NULL) return(NULL);

	for (i = from; i < to; ++i)
	{
		PyObj ob = tup_item(self, i);
		if (ob == NULL) goto error;
		PyTuple_SET_ITEM(rob, i - from, ob);
	}

	return(rob);
error:
	Py_DECREF(rob);
	return(NULL);
}

static PySequenceMethods PyPgHeapTupleAsSequence = {
	tup_length,			/* sq_length */
	NULL,					/* sq_concat */
	NULL,					/* sq_repeat */
	tup_item,			/* sq_item */
	tup_slice,			/* sq_slice */
	NULL,					/* sq_ass_item */
	NULL,					/* sq_ass_slice */
	NULL,					/* sq_contains */
	NULL,					/* sq_inplace_concat */
	NULL,					/* sq_inplace_repeat */
};

static PyObj
tup_subscript(PyObj self, PyObj sub)
{
	TupleDesc td = PyPgHeapTuple_FetchTupleDesc(self);
	PyObj rob = NULL;

	if (PySlice_Check(sub))
	{
		PySliceObject *slice = (PySliceObject *) sub;
		int start, end;

		start = AttrOffset_FromTupleDescAndPyObject(td, slice->start);
		if (PyErr_Occurred()) return(NULL);
		end = AttrOffset_FromTupleDescAndPyObject(td, slice->stop);
		if (PyErr_Occurred()) return(NULL);

		rob = tup_slice(self, start, end);
	}
	else
	{
		int offset;
		offset = AttrOffset_FromTupleDescAndPyObject(td, sub);
		if (PyErr_Occurred()) return(NULL);
		rob = tup_item(self, offset);
	}

	return(rob);
}

static PyMappingMethods PyPgHeapTupleAsMapping = {
	tup_length,				/* mp_length */
	tup_subscript,			/* mp_subscript */
	NULL,						/* mp_ass_subscript */
};

static PyMemberDef PyPgHeapTuple_Members[] = {
	{"TupleDesc", T_OBJECT, offsetof(struct PyPgHeapTuple, ht_desc), RO,
	"the HeapTuple's TupleDesc"},

	{"oid", T_OBJECT, offsetof(struct PyPgHeapTuple, ob_type), RO,
	"the HeapTuple's oid"},
	{"tableoid", T_OBJECT, offsetof(struct PyPgHeapTuple, ob_type), RO,
	"the HeapTuple's tableoid"},

	{"Xmin", T_OBJECT, offsetof(struct PyPgHeapTuple, ob_type), RO,
	"the HeapTuple's Xmin"},
	{"Xmax", T_OBJECT, offsetof(struct PyPgHeapTuple, ob_type), RO,
	"the HeapTuple's Xmax"},
	{"Xvac", T_OBJECT, offsetof(struct PyPgHeapTuple, ob_type), RO,
	"the HeapTuple's Xvac"},

	{"Cmin", T_OBJECT, offsetof(struct PyPgHeapTuple, ob_type), RO,
	"the HeapTuple's Cmin"},
	{"Cmax", T_OBJECT, offsetof(struct PyPgHeapTuple, ob_type), RO,
	"the HeapTuple's Cmax"},

	{"TID", T_OBJECT, offsetof(struct PyPgHeapTuple, ob_type), RO,
	"the HeapTuple's TID"},
	{NULL}
};

static void
tup_dealloc(PyObj self)
{
	HeapTuple ht;
	PyObj ob;

	ht = PyPgHeapTuple_FetchHeapTuple(self);
	if (ht != NULL)
	{
		heap_freetuple(ht);
		PyPgHeapTuple_FixHeapTuple(self, NULL);
	}

	ob = PyPgHeapTuple_FetchPyPgTupleDesc(self);
	if (ob != NULL)
	{
		Py_DECREF(ob);
		PyPgHeapTuple_FixPyPgTupleDesc(self, NULL);
	}

	self->ob_type->tp_free(self);
}

static PyObj
tup_getattro(PyObj self, PyObj attro)
{
	PyObj rob = NULL;
	char *attr = PyString_AS_STRING(attro);

	if (!strcmp(attr, "oid"))
	{
		TupleDesc td = PyPgHeapTuple_FetchTupleDesc(self);
		if (td->tdhasoid && PyPgHeapTuple_HasOid(self))
			rob = PyPg_oid_FromObjectId(PyPgHeapTuple_FetchOid(self));
		else
		{
			Py_INCREF(Py_None);
			rob = Py_None;
		}
	}
	else if (!strcmp(attr, "tableoid"))
		rob = PyPg_oid_FromObjectId(PyPgHeapTuple_FetchTableOid(self));
	else if (!strcmp(attr, "Xmin"))
		rob = PyPg_xid_FromTransactionId(PyPgHeapTuple_FetchXmin(self));
	else if (!strcmp(attr, "Xmax"))
		rob = PyPg_xid_FromTransactionId(PyPgHeapTuple_FetchXmax(self));
	else if (!strcmp(attr, "Xvac"))
		rob = PyPg_xid_FromTransactionId(PyPgHeapTuple_FetchXvac(self));
	else if (!strcmp(attr, "Cmin"))
		rob = PyPg_cid_FromCommandId(PyPgHeapTuple_FetchCmin(self));
	else if (!strcmp(attr, "Cmax"))
		rob = PyPg_cid_FromCommandId(PyPgHeapTuple_FetchCmax(self));
	else if (!strcmp(attr, "TID"))
	{
		ItemPointer to, from;
		from = &PyPgHeapTuple_FetchTID(self);
		to = palloc(SizeOfIptrData);
		if (to)
		{
			ItemPointerCopy(from, to);
			rob = PyPgObject_FromTypeOidAndDatum(TIDOID, PointerGetDatum(to));
		}
	}
	else
		rob = PyObject_GenericGetAttr(self, attro);

	return(rob);
}

static int
tup_compare(PyObj self, PyObj that)
{
	unsigned long i;
	TupleDesc selftupd, thattupd;

	selftupd = PyPgHeapTuple_FetchTupleDesc(self);
	thattupd = PyPgHeapTuple_FetchTupleDesc(that);

	if (selftupd->natts != thattupd->natts)
		return(-1);

	for (i = 0; i < selftupd->natts; ++i)
		if ((selftupd->attrs[i])->atttypid != (thattupd->attrs[i])->atttypid)
			return(-1);

	return(-1);
}

static PyObj
tup_repr(PyObj self)
{
	StringInfoData buf;
	unsigned long i, items;
	PyObj rob = NULL;

	items = PyPgHeapTuple_FetchNatts(self);
	initStringInfo(&buf);
	appendStringInfoChar(&buf, '(');

	for (i = 0; i < items; ++i)
	{
		PyObj ci, obstr;

		ci = self->ob_type->tp_as_sequence->sq_item(self, i);
		if (ci == NULL) goto fail;
		obstr = PyObject_Repr(ci);
		Py_DECREF(ci);
		if (obstr == NULL) goto fail;

		appendStringInfoString(&buf, PyString_AS_STRING(obstr));
		Py_DECREF(obstr);

		if (i+1 < items)
			appendStringInfoString(&buf, ", ");
	}

	appendStringInfoChar(&buf, ')');
	rob = PyString_FromStringAndSize(buf.data, buf.len);
fail:
	pfree(buf.data);
	return(rob);
}

/*
 * This code should reflect the implementation in 8's recordout function.
 * (Some of this code is taken from src/backend/utils/adt/rowtypes.c)
 */
static PyObj
tup_str(PyObj self)
{
	HeapTuple ht = PyPgHeapTuple_FetchHeapTuple(self);
	TupleDesc td = PyPgHeapTuple_FetchTupleDesc(self);
	StringInfoData buf;
	volatile unsigned long i, natts;
	PyObj rob;

	natts = td->natts;
	initStringInfo(&buf);
	appendStringInfoChar(&buf, '(');

	for (i = 0; i < natts; ++i)
	{
		Form_pg_attribute att;
		HeapTuple ttup;
		Form_pg_type ts;
		Oid typoutput, typrelid;
		bool isnull;
		volatile Datum d;
		Oid atttypid;

		att = td->attrs[i];
		if (att->attisdropped)
			continue;

		atttypid = att->atttypid;

		ttup = SearchSysCache(TYPEOID, atttypid, 0, 0, 0);
		ts = TYPESTRUCT(ttup);
		typoutput = ts->typoutput;
		typrelid = ts->typrelid;
		ReleaseSysCache(ttup);

		d = HeapTuple_FetchAttribute(ht, i, td, &isnull);
		if (PyErr_Occurred())
		{
			pfree(buf.data);
			return(NULL);
		}

		if (!isnull)
		{
			bool quote_it = false;
			char * volatile str = NULL, *tmp;

			PgError_TRAP(str = DatumGetCString(OidFunctionCall1(typoutput, d)));
			if (PyErr_Occurred())
			{
				pfree(buf.data);
				return(NULL);
			}
			
			quote_it = (str[0] == '\0');
			for (tmp = str; *tmp; ++tmp)
			{
				char ch = *tmp;
				if (
						ch == '"'
					|| ch == '\\'
					|| ch == '('
					|| ch == ')'
					|| ch == ','
					|| isspace((unsigned char) ch))
				{
					quote_it = true;
					break;
				}
			}

			if (quote_it)
				appendStringInfoChar(&buf, '"');
			for (tmp = str; *tmp; ++tmp)
			{
				char ch = *tmp;

				if (ch == '"' || ch == '\\')
					appendStringInfoChar(&buf, ch);
				appendStringInfoChar(&buf, ch); 
			}
			if (quote_it)
				appendStringInfoChar(&buf, '"');
			
			pfree(str);
		}

		if (i+1 < natts)
			appendStringInfoChar(&buf, ',');
	}
	appendStringInfoChar(&buf, ')');
	appendStringInfoChar(&buf, '\0');

	rob = PyString_FromString(buf.data);
	pfree(buf.data);
	return(rob);
}


const char PyPgHeapTuple_Doc[] = "Postgres HeapTuple interface type";
PyTypeObject PyPgHeapTuple_Type = {
	PyObject_HEAD_INIT(NULL)
	0,										/* ob_size */
	"Postgres.HeapTuple",			/* tp_name */
	sizeof(struct PyPgHeapTuple),	/* tp_basicsize */
	0,										/* tp_itemsize */
	tup_dealloc,						/* tp_dealloc */
	NULL,									/* tp_print */
	NULL,									/* tp_getattr */
	NULL,									/* tp_setattr */
	tup_compare,						/* tp_compare */
	tup_repr,							/* tp_repr */
	NULL,									/* tp_as_number */
	&PyPgHeapTupleAsSequence,		/* tp_as_sequence */
	&PyPgHeapTupleAsMapping,		/* tp_as_mapping */
	NULL,									/* tp_hash */
	NULL,									/* tp_call */
	tup_str,								/* tp_str */
	tup_getattro,						/* tp_getattro */
	NULL,									/* tp_setattro */
	NULL,									/* tp_as_buffer */
	Py_TPFLAGS_DEFAULT,			   /* tp_flags */
	(char *) PyPgHeapTuple_Doc,	/* tp_doc */
	NULL,									/* tp_traverse */
	NULL,									/* tp_clear */
	NULL,									/* tp_richcompare */
	0,										/* tp_weaklistoffset */
	(getiterfunc) PySeqIter_New,	/* tp_iter */
	NULL,									/* tp_iternext */
	NULL,									/* tp_methods */
	PyPgHeapTuple_Members,			/* tp_members */
	NULL,									/* tp_getset */
	NULL,									/* tp_base */
	NULL,									/* tp_dict */
	NULL,									/* tp_descr_get */
	NULL,									/* tp_descr_set */
	0,										/* tp_dictoffset */
	NULL,									/* tp_init */
	NULL,									/* tp_alloc */
	NULL,									/* tp_new */
};

PyObj
PyPgHeapTuple_NEW(PyTypeObject *subtype, PyObj tupd, HeapTuple tup)
{
	PyObj rob;
	HeapTuple ht;

	ht = PyPg_heap_copytuple(tup);
	if (ht == NULL) return(NULL);

	rob = ((PyObj) subtype->tp_alloc(subtype, 0));

	PyPgHeapTuple_FixHeapTuple(rob, ht);
	PyPgHeapTuple_FixPyPgTupleDesc(rob, tupd);
	Py_INCREF(tupd);

	return(rob);
}
/*
 * vim: ts=3:sw=3:noet:
 */
