/* $Id: record.c,v 1.10 2005/12/28 23:54:29 jwp Exp $
 *
 * Copyright 2005, PostgresPy Project.
 * http://python.projects.postgresql.org
 *
 *//*
 * Interface to composite types
 */
#include <setjmp.h>
#include <postgres.h>
#include <access/heapam.h>
#include <access/hash.h>
#include <catalog/pg_type.h>
#include <catalog/pg_proc.h>
#include <catalog/pg_operator.h>
#include <funcapi.h>
#include <lib/stringinfo.h>
#include <nodes/params.h>
#include <parser/parse_func.h>
#include <parser/parse_type.h>
#include <parser/parse_oper.h>
#include <tcop/dest.h>
#include <tcop/tcopprot.h>
#include <utils/array.h>
#include <utils/catcache.h>
#include <utils/datum.h>
#include <utils/geo_decls.h>
#include <utils/palloc.h>
#include <utils/builtins.h>
#include <utils/syscache.h>
#include <utils/typcache.h>
#include <utils/tuplestore.h>
#include <pypg/postgres.h>
#include <pypg/environment.h>

#include <Python.h>
#include <structmember.h>
#include <pypg/python.h>

#include <pypg/externs.h>

#include <pypg/type.h>
#include <pypg/type/object.h>
#include <pypg/type/record.h>
#include <pypg/tupledesc.h>
#include <pypg/heaptuple.h>
#include <pypg/error.h>

static int
rec_length(PyObj self)
{
	return(PyPgType_FetchTupleDesc(self->ob_type)->natts);
}

static PyObj
rec_item(PyObj self, int attnum)
{
	bool isnull = false;
	TupleDesc td;
	HeapTupleData ht;
	int natts;
	Datum datum = 0;
	PyObj atttyp, rob;

	td = PyPgType_FetchTupleDesc(self->ob_type);
	natts = td->natts;

	if (attnum >= natts)
	{
		PyErr_Format(PyExc_IndexError,
				"index %d out of range %d", attnum, natts);
		return(NULL);
	}

	atttyp = PyPgType_FromOid(td->attrs[attnum]->atttypid);
	if (atttyp == NULL) return(NULL);

	ht.t_tableOid = InvalidOid;
	ht.t_data = DatumGetHeapTupleHeader(PyPgObject_FetchDatum(self));
	ht.t_len = HeapTupleHeaderGetDatumLength(ht.t_data);

	PgError_TRAP(datum = fastgetattr(&ht, attnum + 1, td, &isnull));
	if (PyErr_Occurred()) return(NULL);

	if (isnull)
	{
		rob = Py_None;
		Py_INCREF(rob);
	}
	else
		rob = PyPgObject_FromPyPgTypeAndDatum(atttyp, datum);

	return(rob);
}

static PyObj
rec_slice(PyObj self, int from, int to)
{
	PyObj rob;
	int i;

	if (to == INT_MAX) to = PyPgType_FetchTupleDesc(self->ob_type)->natts;

	rob = PyTuple_New(to - from);
	if (rob == NULL) return(NULL);

	for (i = from; i < to; ++i)
	{
		PyObj ob = rec_item(self, i);
		if (ob == NULL) goto error;
		PyTuple_SET_ITEM(rob, i - from, ob);
	}

	return(rob);
error:
	Py_DECREF(rob);
	return(NULL);
}

static PySequenceMethods record_as_sequence = {
	rec_length,			/* sq_length */
	NULL,					/* sq_concat */
	NULL,					/* sq_repeat */
	rec_item,			/* sq_item */
	rec_slice,			/* sq_slice */
	NULL,					/* sq_ass_item */
	NULL,					/* sq_ass_slice */
	NULL,					/* sq_contains */
};

static PyObj
rec_subscript(PyObj self, PyObj sub)
{
	TupleDesc td = PyPgType_FetchTupleDesc(self->ob_type);
	PyObj rob = NULL;

	if (PySlice_Check(sub))
	{
		PySliceObject *slice = (PySliceObject *) sub;
		int start, end;

		start = AttrOffset_FromTupleDescAndPyObject(td, slice->start);
		if (PyErr_Occurred()) return(NULL);
		end = AttrOffset_FromTupleDescAndPyObject(td, slice->stop);
		if (PyErr_Occurred()) return(NULL);

		rob = rec_slice(self, start, end);
	}
	else
	{
		int offset;
		offset = AttrOffset_FromTupleDescAndPyObject(td, sub);
		if (PyErr_Occurred()) return(NULL);
		rob = rec_item(self, offset);
	}

	return(rob);
}

static PyMappingMethods record_as_mapping = {
	rec_length,				/* mp_length */
	rec_subscript,			/* mp_subscript */
	NULL,						/* mp_ass_subscript */
};

static PyObj
rec_new(PyTypeObject *subtype, PyObj args, PyObj kw)
{
	PyObj ob, rob = NULL;

	if ((PyObj) subtype == (PyObj) &PyPg_record_Type)
	{
		PyErr_SetString(PyExc_TypeError,
			"cannot instantiate anonymous records");
		return(NULL);
	}

	if (!PyObject_IsSubclass((PyObj) subtype, (PyObj) &PyPg_record_Type))
	{
		PyErr_Format(PyExc_TypeError,
			"subtype must be a Postgres.types.record, not a '%s'", 
			subtype->ob_type->tp_name
		);
		return(NULL);
	}

	if (!PyPgArg_ParseTypeSource(args, kw, &ob))
		return(NULL);

	if (ob->ob_type == subtype)
	{
		Py_INCREF(ob);
		return(ob);
	}

	if (PyString_Check(ob))
		rob = PyPgObject_typinput((PyObj) subtype, ob);
	else
	{
		TupleDesc td = PyPgType_FetchTupleDesc(subtype);
		HeapTuple ht;

		ht = HeapTuple_FromTupleDescAndIterable(td, ob);
		if (ht == NULL) return(NULL);

		if (td->tdtypeid != RECORDOID)
			HeapTupleHeaderSetTypMod(ht->t_data, -1);
		else
			HeapTupleHeaderSetTypMod(ht->t_data, td->tdtypmod);
		HeapTupleHeaderSetTypeId(ht->t_data, PyPgType_FetchOid(subtype));
		HeapTupleHeaderSetDatumLength(ht->t_data, ht->t_len + VARHDRSZ);
		rob = PyPgObject_New(subtype, PointerGetDatum(ht->t_data));
		heap_freetuple(ht);
	}

	return(rob);
}

PyDoc_STRVAR(PyPg_record_Type_Doc, "record interface type");

PyPgTypeObject PyPg_record_Type = {{
	PyObject_HEAD_INIT(&PyPgType_Type)
	0,												/* ob_size */
	"record",									/* tp_name */
	sizeof(struct PyPgObject),				/* tp_basicsize */
	0,												/* tp_itemsize */
	NULL,											/* tp_dealloc */
	NULL,											/* tp_print */
	NULL,											/* tp_getattr */
	NULL,											/* tp_setattr */
	NULL,											/* tp_compare */
	NULL,											/* tp_repr */
	NULL,											/* tp_as_number */
	&record_as_sequence,						/* tp_as_sequence */
	&record_as_mapping,						/* tp_as_mapping */
	NULL,											/* tp_hash */
	NULL,											/* tp_call */
	NULL,											/* tp_str */
	NULL,											/* tp_getattro */
	NULL,											/* tp_setattro */
	NULL,											/* tp_as_buffer */
	PyPg_TPFLAGS_DEFAULT|
	Py_TPFLAGS_BASETYPE,						/* tp_flags */
	PyPg_record_Type_Doc,					/* tp_doc */
	NULL,											/* tp_traverse */
	NULL,											/* tp_clear */
	NULL,											/* tp_richcompare */
	0,												/* tp_weaklistoffset */
	NULL,											/* tp_iter */
	NULL,											/* tp_iternext */
	NULL,											/* tp_methods */
	NULL,											/* tp_members */
	NULL,											/* tp_getset */
	(PyTypeObject *) &PyPgObject_Type,	/* tp_base */
	NULL,											/* tp_dict */
	NULL,											/* tp_descr_get */
	NULL,											/* tp_descr_set */
	0,												/* tp_dictoffset */
	NULL,											/* tp_init */
	NULL,											/* tp_alloc */
	rec_new,										/* tp_new */
	NULL,											/* tp_free */
},
};

PyDoc_STRVAR(PyPg_attribute_Type_Doc, "pg_attribute interface type");

PyPgTypeObject PyPg_attribute_Type = {{
	PyObject_HEAD_INIT(&PyPgType_Type)
	0,												/* ob_size */
	"attribute",								/* tp_name */
	sizeof(struct PyPg_record),			/* tp_basicsize */
	0,												/* tp_itemsize */
	NULL,											/* tp_dealloc */
	NULL,											/* tp_print */
	NULL,											/* tp_getattr */
	NULL,											/* tp_setattr */
	NULL,											/* tp_compare */
	NULL,											/* tp_repr */
	NULL,											/* tp_as_number */
	NULL,											/* tp_as_sequence */
	NULL,											/* tp_as_mapping */
	NULL,											/* tp_hash */
	NULL,											/* tp_call */
	NULL,											/* tp_str */
	NULL,											/* tp_getattro */
	NULL,											/* tp_setattro */
	NULL,											/* tp_as_buffer */
	PyPg_TPFLAGS_DEFAULT,					/* tp_flags */
	PyPg_attribute_Type_Doc,				/* tp_doc */
	NULL,											/* tp_traverse */
	NULL,											/* tp_clear */
	NULL,											/* tp_richcompare */
	0,												/* tp_weaklistoffset */
	NULL,											/* tp_iter */
	NULL,											/* tp_iternext */
	NULL,											/* tp_methods */
	NULL,											/* tp_members */
	NULL,											/* tp_getset */
	(PyTypeObject *) &PyPg_record_Type,	/* tp_base */
	NULL,											/* tp_dict */
	NULL,											/* tp_descr_get */
	NULL,											/* tp_descr_set */
	0,												/* tp_dictoffset */
	NULL,											/* tp_init */
	NULL,											/* tp_alloc */
	NULL,											/* tp_new */
	NULL,											/* tp_free */
},
};

PyObj
PyPg_record_Type_FromTupleDesc(TupleDesc td)
{
	PyObj bases, nbd, rob;
	PyPgTypeInfo ti;

	bases = PyTuple_New(1);
	if (bases == NULL) return(NULL);

	PyTuple_SET_ITEM(bases, 0, (PyObj) &PyPg_record_Type);
	Py_INCREF((PyObj) &PyPg_record_Type);
	nbd = PyTuple_New(3);
	if (nbd == NULL)
	{
		Py_DECREF(bases);
		return(NULL);
	}
	PyTuple_SET_ITEM(nbd, 0, PyString_FromString("custom record"));
	PyTuple_SET_ITEM(nbd, 1, bases);
	PyTuple_SET_ITEM(nbd, 2, PyDict_New());
	rob = PyType_Type.tp_new(((PyObj) &PyPg_record_Type)->ob_type, nbd, NULL);
	Py_DECREF(nbd);

	/* XXX: Use a typmod cache? */
	if (rob != NULL)
	{
		if (_PyPgType_Ready((PyPgTypeObject *) rob, RECORDOID) < 0)
		{
			Py_DECREF(rob);
			rob = NULL;
		}
	}

	ti = PyPgTypeInfoPointer(rob);
	if (ti->typ_td != NULL)
	{
		Py_DECREF(ti->typ_td);
	}
	ti->typ_td = PyPgTupleDesc_New(td);
	BlessTupleDesc(PyPgTupleDesc_FetchTupleDesc(ti->typ_td));

	return(rob);
}
/*
 * vim: ts=3:sw=3:noet:
 */
