/*-------------------------------------------------------------------------
 *
 * pl_funcs.c		- Misc functions for the PL/pgSQL
 *			  procedural language
 *
 * Portions Copyright (c) 1996-2006, PostgreSQL Global Development Group
 * Portions Copyright (c) 1994, Regents of the University of California
 *
 *
 * IDENTIFICATION
 *	  $PostgreSQL: pgsql/src/pl/plpgpsm/src/pl_funcs.c,v 1.55 2006/09/22 21:39:58 tgl Exp $
 *
 *-------------------------------------------------------------------------
 */

#include "plpgpsm.h"

#include <ctype.h>

#include "parser/scansup.h"


/* ----------
 * Local variables for the namestack handling
 * ----------
 */
static PLpgPSM_ns *ns_current = NULL;
static bool ns_localmode = false;


/* ----------
 * plpgpsm_dstring_init			Dynamic string initialization
 * ----------
 */
void
plpgpsm_dstring_init(PLpgPSM_dstring *ds)
{
	ds->value = palloc(ds->alloc = 512);
	ds->used = 1;
	ds->value[0] = '\0';
}


/* ----------
 * plpgpsm_dstring_free			Dynamic string destruction
 * ----------
 */
void
plpgpsm_dstring_free(PLpgPSM_dstring *ds)
{
	pfree(ds->value);
}

static void
plpgpsm_dstring_expand(PLpgPSM_dstring *ds, int needed)
{
	/* Don't allow truncating the string */
	Assert(needed > ds->alloc);
	Assert(ds->used <= ds->alloc);

	/* Might have to double more than once, if needed is large */
	do
	{
		ds->alloc *= 2;
	} while (needed > ds->alloc);
	ds->value = repalloc(ds->value, ds->alloc);
}

/* ----------
 * plpgpsm_dstring_append		Dynamic string extending
 * ----------
 */
void
plpgpsm_dstring_append(PLpgPSM_dstring *ds, const char *str)
{
	int			len = strlen(str);
	int			needed = ds->used + len;

	if (needed > ds->alloc)
		plpgpsm_dstring_expand(ds, needed);

	memcpy(&(ds->value[ds->used - 1]), str, len);
	ds->used += len;
	ds->value[ds->used - 1] = '\0';
}

/* ----------
 * plpgpsm_dstring_append_char	Append a single character
 *								to a dynamic string
 * ----------
 */
void
plpgpsm_dstring_append_char(PLpgPSM_dstring *ds, char c)
{
	if (ds->used == ds->alloc)
		plpgpsm_dstring_expand(ds, ds->used + 1);

	ds->value[ds->used - 1] = c;
	ds->value[ds->used] = '\0';
	ds->used++;
}


/* ----------
 * plpgpsm_dstring_get			Dynamic string get value
 * ----------
 */
char *
plpgpsm_dstring_get(PLpgPSM_dstring *ds)
{
	return ds->value;
}


/* ----------
 * plpgpsm_ns_init			Initialize the namestack
 * ----------
 */
void
plpgpsm_ns_init(void)
{
	ns_current = NULL;
	ns_localmode = false;
}

/* ----------
 * plpgpsm_ns_setlocal			Tell plpgpsm_ns_lookup whether to
 *					look into the current level only.
 *
 * This is a crock, but in the current design we need it because scan.l
 * initiates name lookup, and the scanner does not know whether we are
 * examining a name being declared in a DECLARE section.  For that case
 * we only want to know if there is a conflicting name earlier in the
 * same DECLARE section.  So the grammar must temporarily set local mode
 * before scanning decl_varnames.
 * ----------
 */
bool
plpgpsm_ns_setlocal(bool flag)
{
	bool		oldstate;

	oldstate = ns_localmode;
	ns_localmode = flag;
	return oldstate;
}


/* ----------
 * plpgpsm_ns_push			Enter a new namestack level
 * ----------
 */
void
plpgpsm_ns_push(const char *label)
{
	PLpgPSM_ns *new;

	if (label == NULL)
		label = "";

	new = palloc0(sizeof(PLpgPSM_ns));
	new->upper = ns_current;
	ns_current = new;

	plpgpsm_ns_additem(PLPGPSM_NSTYPE_LABEL, 0, label);
}


/* ----------
 * plpgpsm_ns_pop			Return to the previous level
 * ----------
 */
void
plpgpsm_ns_pop(void)
{
	int			i;
	PLpgPSM_ns *old;

	old = ns_current;
	ns_current = old->upper;

	for (i = 0; i < old->items_used; i++)
		pfree(old->items[i]);
	pfree(old->items);
	pfree(old);
}


/* ----------
 * plpgpsm_ns_additem			Add an item to the current
 *					namestack level
 * ----------
 */
void
plpgpsm_ns_additem(int itemtype, int itemno, const char *name)
{
	PLpgPSM_ns *ns = ns_current;
	PLpgPSM_nsitem *nse;

	Assert(name != NULL);

	if (ns->items_used == ns->items_alloc)
	{
		if (ns->items_alloc == 0)
		{
			ns->items_alloc = 32;
			ns->items = palloc(sizeof(PLpgPSM_nsitem *) * ns->items_alloc);
		}
		else
		{
			ns->items_alloc *= 2;
			ns->items = repalloc(ns->items,
								 sizeof(PLpgPSM_nsitem *) * ns->items_alloc);
		}
	}

	nse = palloc(sizeof(PLpgPSM_nsitem) + strlen(name));
	nse->itemtype = itemtype;
	nse->itemno = itemno;
	strcpy(nse->name, name);
	ns->items[ns->items_used++] = nse;
}


/* ----------
 * plpgpsm_ns_lookup			Lookup an identifier in the namestack
 *
 * Note that this only searches for variables, not labels.
 *
 * name1 must be non-NULL.  Pass NULL for name2 and/or name3 if parsing a name
 * with fewer than three components.
 *
 * If names_used isn't NULL, *names_used receives the number of names
 * matched: 0 if no match, 1 if name1 matched an unqualified variable name,
 * 2 if name1 and name2 matched a block label + variable name.
 *
 * Note that name3 is never directly matched to anything.  However, if it
 * isn't NULL, we will disregard qualified matches to scalar variables.
 * Similarly, if name2 isn't NULL, we disregard unqualified matches to
 * scalar variables.
 * ----------
 */
PLpgPSM_nsitem *
plpgpsm_ns_lookup(const char *name1, const char *name2, const char *name3,
				  int *names_used)
{
	PLpgPSM_ns *ns;
	int			i;

	/* Scan each level of the namestack */
	for (ns = ns_current; ns != NULL; ns = ns->upper)
	{
		/* Check for unqualified match to variable name */
		for (i = 1; i < ns->items_used; i++)
		{
			PLpgPSM_nsitem *nsitem = ns->items[i];

			if (strcmp(nsitem->name, name1) == 0)
			{
				if (name2 == NULL ||
					nsitem->itemtype != PLPGPSM_NSTYPE_VAR)
				{
					if (names_used)
						*names_used = 1;
					return nsitem;
				}
			}
		}

		/* Check for qualified match to variable name */
		if (name2 != NULL &&
			strcmp(ns->items[0]->name, name1) == 0)
		{
			for (i = 1; i < ns->items_used; i++)
			{
				PLpgPSM_nsitem *nsitem = ns->items[i];

				if (strcmp(nsitem->name, name2) == 0)
				{
					if (name3 == NULL ||
						nsitem->itemtype != PLPGPSM_NSTYPE_VAR)
					{
						if (names_used)
							*names_used = 2;
						return nsitem;
					}
				}
			}
		}

		if (ns_localmode)
			break;				/* do not look into upper levels */
	}

	/* This is just to suppress possibly-uninitialized-variable warnings */
	if (names_used)
		*names_used = 0;
	return NULL;				/* No match found */
}


/* ----------
 * plpgpsm_ns_lookup_label		Lookup a label in the namestack
 * ----------
 */
PLpgPSM_nsitem *
plpgpsm_ns_lookup_label(const char *name)
{
	PLpgPSM_ns *ns;

	for (ns = ns_current; ns != NULL; ns = ns->upper)
	{
		if (strcmp(ns->items[0]->name, name) == 0)
			return ns->items[0];
	}

	return NULL;				/* label not found */
}


/* ----------
 * plpgpsm_ns_rename			Rename a namespace entry
 * ----------
 */
void
plpgpsm_ns_rename(char *oldname, char *newname)
{
	PLpgPSM_ns *ns;
	PLpgPSM_nsitem *newitem;
	int			i;

	/*
	 * Lookup name in the namestack; do the lookup in the current namespace
	 * only.
	 */
	for (ns = ns_current; ns != NULL; ns = ns->upper)
	{
		for (i = 1; i < ns->items_used; i++)
		{
			if (strcmp(ns->items[i]->name, oldname) == 0)
			{
				newitem = palloc(sizeof(PLpgPSM_nsitem) + strlen(newname));
				newitem->itemtype = ns->items[i]->itemtype;
				newitem->itemno = ns->items[i]->itemno;
				strcpy(newitem->name, newname);

				pfree(oldname);
				pfree(newname);

				pfree(ns->items[i]);
				ns->items[i] = newitem;
				return;
			}
		}
	}

	ereport(ERROR,
			(errcode(ERRCODE_UNDEFINED_OBJECT),
			 errmsg("there is no variable \"%s\" in the current block",
					oldname)));
}


/* ----------
 * plpgpsm_convert_ident
 *
 * Convert a possibly-qualified identifier to internal form: handle
 * double quotes, translate to lower case where not inside quotes,
 * truncate to NAMEDATALEN.
 *
 * There may be several identifiers separated by dots and optional
 * whitespace.	Each one is converted to a separate palloc'd string.
 * The caller passes the expected number of identifiers, as well as
 * a char* array to hold them.	It is an error if we find the wrong
 * number of identifiers (cf grammar processing of fori_varname).
 *
 * NOTE: the input string has already been accepted by the flex lexer,
 * so we don't need a heckuva lot of error checking here.
 * ----------
 */
void
plpgpsm_convert_ident(const char *s, char **output, int numidents)
{
	const char *sstart = s;
	int			identctr = 0;

	/* Outer loop over identifiers */
	while (*s)
	{
		char	   *curident;
		char	   *cp;

		/* Process current identifier */

		if (*s == '"')
		{
			/* Quoted identifier: copy, collapsing out doubled quotes */

			curident = palloc(strlen(s) + 1);	/* surely enough room */
			cp = curident;
			s++;
			while (*s)
			{
				if (*s == '"')
				{
					if (s[1] != '"')
						break;
					s++;
				}
				*cp++ = *s++;
			}
			if (*s != '"')		/* should not happen if lexer checked */
				ereport(ERROR,
						(errcode(ERRCODE_SYNTAX_ERROR),
						 errmsg("unterminated \" in name: %s", sstart)));
			s++;
			*cp = '\0';
			/* Truncate to NAMEDATALEN */
			truncate_identifier(curident, cp - curident, false);
		}
		else
		{
			/* Normal identifier: extends till dot or whitespace */
			const char *thisstart = s;

			while (*s && *s != '.' && !scanner_isspace(*s))
				s++;
			/* Downcase and truncate to NAMEDATALEN */
			curident = downcase_truncate_identifier(thisstart, s - thisstart,
													false);
		}

		/* Pass ident to caller */
		if (identctr < numidents)
			output[identctr++] = curident;
		else
			ereport(ERROR,
					(errcode(ERRCODE_SYNTAX_ERROR),
					 errmsg("qualified identifier cannot be used here: %s",
							sstart)));

		/* If not done, skip whitespace, dot, whitespace */
		if (*s)
		{
			while (*s && scanner_isspace(*s))
				s++;
			if (*s++ != '.')
				elog(ERROR, "expected dot between identifiers: %s", sstart);
			while (*s && scanner_isspace(*s))
				s++;
			if (*s == '\0')
				elog(ERROR, "expected another identifier: %s", sstart);
		}
	}

	if (identctr != numidents)
		elog(ERROR, "improperly qualified identifier: %s",
			 sstart);
}


/*
 * Statement type as a string, for use in error messages etc.
 */
const char *
plpgpsm_stmt_typename(PLpgPSM_stmt *stmt)
{
	switch (stmt->cmd_type)
	{
		case PLPGPSM_STMT_BLOCK:
			return _("BEGIN END");
		case PLPGPSM_STMT_ASSIGN:
			return ("ASSIGNMENT");
		case PLPGPSM_STMT_IF:
			return "IF";
		case PLPGPSM_STMT_LOOP:
			return "LOOP";
		case PLPGPSM_STMT_FORS:
			return "FOR";
		case PLPGPSM_STMT_LEAVE:
			return "LEAVE";
		case PLPGPSM_STMT_RETURN:
			return "RETURN";
		case PLPGPSM_STMT_SIGNAL:
			return "SIGNAL";
		case PLPGPSM_STMT_SQL:
			return "SQL";
		case PLPGPSM_STMT_EXECUTE:
			return "EXECUTE";
		case PLPGPSM_STMT_GETDIAG:
			return "GET DIAGNOSTIC";
		case PLPGPSM_STMT_OPEN:
			return "OPEN";
		case PLPGPSM_STMT_FETCH:
			return "FETCH";
		case PLPGPSM_STMT_CLOSE:
			return "CLOSE";
		case PLPGPSM_STMT_CALL:
			return "CALL";
		case PLPGPSM_STMT_CASE:
			return "CASE";
		case PLPGPSM_STMT_PRINT:
			return "PRINT";
			
	}

	return "unknown";
}


/**********************************************************************
 * Debug functions for analyzing the compiled code
 **********************************************************************/
static int	dump_indent;

static void dump_ind();
static void dump_stmt(PLpgPSM_stmt *stmt);
static void dump_block(PLpgPSM_stmt_block *block);
static void dump_assign(PLpgPSM_stmt_assign *stmt);
static void dump_if(PLpgPSM_stmt_if *stmt);
static void dump_loop(PLpgPSM_stmt_loop *stmt);
static void dump_fors(PLpgPSM_stmt_fors *stmt);
static void dump_leave(PLpgPSM_stmt_leave *stmt);
static void dump_return(PLpgPSM_stmt_return *stmt);
static void dump_signal(PLpgPSM_stmt_signal *stmt);
static void dump_sql(PLpgPSM_stmt_sql *stmt);
static void dump_execute(PLpgPSM_stmt_execute *stmt);
static void dump_getdiag(PLpgPSM_stmt_getdiag *stmt);
static void dump_open(PLpgPSM_stmt_open *stmt);
static void dump_fetch(PLpgPSM_stmt_fetch *stmt);
static void dump_close(PLpgPSM_stmt_close *stmt);
static void dump_call(PLpgPSM_stmt_call *stmt);
static void dump_expr(PLpgPSM_expr *expr);
static void dump_print(PLpgPSM_stmt_print *print);
static void dump_case(PLpgPSM_stmt_case *stmt);
static void dump_prepare(PLpgPSM_stmt_prepare *prepare);
static void dump_cursor_direction(PLpgPSM_stmt_fetch *stmt);


static void
dump_ind(void)
{
	int			i;

	for (i = 0; i < dump_indent; i++)
		printf(" ");
}

static void
dump_stmt(PLpgPSM_stmt *stmt)
{
	printf("%3d:", stmt->lineno);
	switch (stmt->cmd_type)
	{
		case PLPGPSM_STMT_BLOCK:
			dump_block((PLpgPSM_stmt_block *) stmt);
			break;
		case PLPGPSM_STMT_ASSIGN:
			dump_assign((PLpgPSM_stmt_assign *) stmt);
			break;
		case PLPGPSM_STMT_IF:
			dump_if((PLpgPSM_stmt_if *) stmt);
			break;
		case PLPGPSM_STMT_LOOP:
			dump_loop((PLpgPSM_stmt_loop *) stmt);
			break;
		case PLPGPSM_STMT_FORS:
			dump_fors((PLpgPSM_stmt_fors *) stmt);
			break;
		case PLPGPSM_STMT_LEAVE:
			dump_leave((PLpgPSM_stmt_leave *) stmt);
			break;
		case PLPGPSM_STMT_RETURN:
			dump_return((PLpgPSM_stmt_return *) stmt);
			break;
		case PLPGPSM_STMT_SIGNAL:
			dump_signal((PLpgPSM_stmt_signal *) stmt);
			break;
		case PLPGPSM_STMT_SQL:
			dump_sql((PLpgPSM_stmt_sql *) stmt);
			break;
		case PLPGPSM_STMT_EXECUTE:
			dump_execute((PLpgPSM_stmt_execute *) stmt);
			break;
		case PLPGPSM_STMT_GETDIAG:
			dump_getdiag((PLpgPSM_stmt_getdiag *) stmt);
			break;
		case PLPGPSM_STMT_OPEN:
			dump_open((PLpgPSM_stmt_open *) stmt);
			break;
		case PLPGPSM_STMT_FETCH:
			dump_fetch((PLpgPSM_stmt_fetch *) stmt);
			break;
		case PLPGPSM_STMT_CLOSE:
			dump_close((PLpgPSM_stmt_close *) stmt);
			break;
		case PLPGPSM_STMT_CALL:
			dump_call((PLpgPSM_stmt_call *) stmt);
			break;
		case PLPGPSM_STMT_PRINT:
			dump_print((PLpgPSM_stmt_print *) stmt);
			break;
		case PLPGPSM_STMT_CASE:
			dump_case((PLpgPSM_stmt_case *) stmt);
			break;
		case PLPGPSM_STMT_PREPARE:
			dump_prepare((PLpgPSM_stmt_prepare *) stmt);
			break;

		default:
			elog(ERROR, "unrecognized cmd_type: %d", stmt->cmd_type);
			break;
	}
}

static void
dump_stmts(List *stmts)
{
	ListCell   *s;

	dump_indent += 2;
	foreach(s, stmts)
		dump_stmt((PLpgPSM_stmt *) lfirst(s));
	dump_indent -= 2;
}

static void
dump_block(PLpgPSM_stmt_block *block)
{
	char	   *name;

	if (block->label == NULL)
		name = "*unnamed*";
	else
		name = block->label;

	dump_ind();
	printf("%s:\n", name);
	dump_ind();
	printf("    BLOCK %s\n", block->atomic ? "ATOMIC ":"");

	if (block->exceptions)
	{
		ListCell   *e;

		dump_indent += 2;

		foreach(e, block->exceptions->exc_list)
		{
			PLpgPSM_exception *exc = (PLpgPSM_exception *) lfirst(e);
			PLpgPSM_condition *cond;

			dump_ind();
			switch (exc->handler_type)
			{
				case PLPGPSM_HDL_CONTINUE:
				    printf("    CONTINUE ");
				    break;

				case PLPGPSM_HDL_EXIT:
				    printf("    EXIT ");
				    break;
				
				case PLPGPSM_HDL_UNDO:
				    printf("    UNDO ");
				    break;
			}
			printf("EXCEPTION HANDLER FOR ");
			for (cond = exc->conditions; cond; cond = cond->next)
			{
				if (cond != exc->conditions)
					printf(" OR ");
				printf("%s", cond->condname);
			}
			printf(" \n");
			dump_indent += 2;
			dump_stmt(exc->action);
			dump_indent -= 2;
		}

		dump_indent -= 2;
	}

	dump_stmts(block->body);

	dump_ind();
	printf("    END %s\n", name);
}

static void
dump_assign(PLpgPSM_stmt_assign *stmt)
{
	if (!stmt->varname)
	{
		ListCell   *lc;

		dump_ind();
		printf("ASSIGN\n"); 
		dump_indent += 2;
		foreach(lc, stmt->items)
		{
			PLpgPSM_assign_item *ai = (PLpgPSM_assign_item *) lfirst(lc);
			dump_ind();
			printf("    Assign var %d := ", ai->varno);
			dump_expr(ai->expr);
			printf("\n");
		}	 
		dump_indent -= 2;
	}
	else
	{
		dump_ind();
		printf("ASSIGN SYSTEM VARIABLE %s := ", stmt->varname);
		dump_expr(stmt->expr);
		printf("\n");
	}
}

static void
dump_if(PLpgPSM_stmt_if *stmt)
{
	dump_ind();
	printf("IF ");
	dump_expr(stmt->cond);
	printf(" THEN\n");

	dump_stmts(stmt->true_body);

	if (stmt->false_body != NIL)
	{
		dump_ind();
		printf("    ELSE\n");
		dump_stmts(stmt->false_body);
	}

	dump_ind();
	printf("    ENDIF\n");
}

static void
dump_case(PLpgPSM_stmt_case *stmt)
{
	ListCell	*l;

	dump_ind();
	printf("CASE %d ", stmt->t_varno);
	if (stmt->t_expr)
		dump_expr(stmt->t_expr);
	printf("\n");
	dump_indent += 6;
	foreach(l, stmt->case_when_list)
	{
		PLpgPSM_case_when *cwt = (PLpgPSM_case_when *) lfirst(l);

		dump_ind();
		printf("WHEN ");
		dump_expr(cwt->expr);
		printf("\n");
		dump_ind();
		printf("THEN\n");
		dump_indent += 2;
		dump_stmts(cwt->stmts);
		dump_indent -= 2;
	}
	if (stmt->have_else)
	{
		dump_ind();
		printf("ELSE\n");
		dump_indent += 2;
		dump_stmts(stmt->else_stmts);
		dump_indent -= 2;
	}
	dump_indent -= 6;
	dump_ind();
	printf("    ENDCASE\n");
}



static void
dump_loop(PLpgPSM_stmt_loop *stmt)
{
	char	   *name;

	if (stmt->label == NULL)
		name = "*unnamed*";
	else
		name = stmt->label;

	dump_ind();
	printf("%s:\n", name);
	dump_ind();
	
	switch(stmt->ctype)
	{
		case PLPGPSM_CYCLE_LOOP:
			printf("    LOOP\n");
			dump_stmts(stmt->body);
			dump_ind();
			printf("    ENDLOOP %s\n", name);
			break;

		case PLPGPSM_CYCLE_WHILE:
			printf("    WHILE ");
			dump_expr(stmt->cond);
			printf(" DO\n");
			dump_stmts(stmt->body);
			dump_ind();
			printf("    ENDWHILE %s\n", name);
			break;

		case PLPGPSM_CYCLE_REPEAT:
			printf("    REPEAT\n");
			dump_stmts(stmt->body);
			dump_ind();
			printf("    UNTIL\n");
			dump_ind();
			printf("      ");
			dump_expr(stmt->cond);
			printf("\n");
			dump_ind();
			printf("    ENDREPEAT %s\n", name);
			break;
	}
}


static void
dump_fors(PLpgPSM_stmt_fors *stmt)
{
	char	   *name;

	if (stmt->label == NULL)
		name = "*unnamed*";
	else
		name = stmt->label;

	dump_ind();
	printf("%s: FORS %s ", name, (stmt->rec != NULL) ? stmt->rec->refname : stmt->row->refname);
	dump_expr(stmt->query);
	printf("\n");

	dump_stmts(stmt->body);

	dump_ind();
	printf("    ENDFORS %s\n", name);
}

static void
dump_open(PLpgPSM_stmt_open *stmt)
{
	dump_ind();
	printf("OPEN curvar=%d\n", stmt->curvar);

	dump_indent += 2;

	if (stmt->args)
	{
		int i;

		dump_ind();
		printf("    USING {");
		
		for (i = 0; i < stmt->args->nparams; i++)
		{
			if (i > 0) 
				printf(", ");
			printf("$%d=%d", i + 1, stmt->args->params[i]);
		}
		printf("}\n");		
    	}

	dump_indent -= 2;

}


static void
dump_fetch(PLpgPSM_stmt_fetch *stmt)
{
	dump_ind();
	
	if (!stmt->is_move)
	{
		printf("FETCH curvar=%d\n", stmt->curvar);
		dump_cursor_direction(stmt);

		dump_indent += 2;
		if (stmt->rec != NULL)
		{
			dump_ind();
			printf("    target = %d %s\n", stmt->rec->dno, stmt->rec->refname);
		}
		if (stmt->row != NULL)
		{
			dump_ind();
			printf("    target = %d %s\n", stmt->row->dno, stmt->row->refname);
		}
		dump_indent -= 2;
	}
	else
	{
		printf("MOVE curvar=%d\n", stmt->curvar);
		dump_cursor_direction(stmt);
	}
}


static void
dump_close(PLpgPSM_stmt_close *stmt)
{
	dump_ind();
	printf("CLOSE curvar=%d\n", stmt->curvar);
}


static void
dump_call(PLpgPSM_stmt_call *stmt)
{
	dump_ind();
	printf("CALL expr = ");
	dump_expr(stmt->expr);
	printf("\n");
}


static void
dump_leave(PLpgPSM_stmt_leave *stmt)
{
	dump_ind();
	printf("%s label='%s'",
		   stmt->is_leave ? "LEAVE" : "ITERATE", stmt->label);
	printf("\n");
}


static void
dump_return(PLpgPSM_stmt_return *stmt)
{
	dump_ind();
	printf("RETURN ");
	if (stmt->is_tblexpr)
		printf("TABLE ");
	if (stmt->retvarno >= 0)
		printf("variable %d", stmt->retvarno);
	else if (stmt->expr != NULL)
		dump_expr(stmt->expr);
	else
		printf("NULL");
	printf("\n");
}


static void 
dump_print(PLpgPSM_stmt_print *stmt)
{
	ListCell   *lc;
	int			i = 0;

	dump_ind();
	printf("PRINT\n"); 
	dump_indent += 2;
	foreach(lc, stmt->params)
	{
		dump_ind();
		printf("    parameter %d: ", i++);
		dump_expr((PLpgPSM_expr *) lfirst(lc));
		printf("\n");
	} 
	dump_indent -= 2;
}


static void
dump_signal(PLpgPSM_stmt_signal *stmt)
{
	char *condname = "";
	
	dump_ind();
	
	if (stmt->condname && strncmp(stmt->condname, "SQLSTATE ", 9) != 0)
	    condname = stmt->condname;

	printf("SIGNAL %s SQLSTATE %s\n", condname, unpack_sql_state(stmt->sqlerrstate)); 
	dump_indent += 2;
	
	if (stmt->message)
	{
		dump_ind();
		printf("    Message = ");
		dump_expr(stmt->message);
		printf("\n");	
	}

	if (stmt->hint)
	{
		dump_ind();
		printf("    Hint = ");
		dump_expr(stmt->hint);
		printf("\n");	
	}

	if (stmt->detail)
	{
		dump_ind();
		printf("    Detail = ");
		dump_expr(stmt->detail);
		printf("\n");	
	}

	dump_indent -= 2;
}


static void
dump_sql(PLpgPSM_stmt_sql *stmt)
{
	dump_ind();
	printf("SQL ");
	dump_expr(stmt->sqlstmt);
	printf("\n");

	dump_indent += 2;
	if (stmt->rec != NULL)
	{
		dump_ind();
		printf("    INTO%s target = %d %s\n",
			   stmt->strict ? " STRICT" : "",
			   stmt->rec->dno, stmt->rec->refname);
	}
	if (stmt->row != NULL)
	{
		dump_ind();
		printf("    INTO%s target = %d %s\n",
			   stmt->strict ? " STRICT" : "",
			   stmt->row->dno, stmt->row->refname);
	}
	dump_indent -= 2;
	
}

static void 
dump_prepare(PLpgPSM_stmt_prepare *prepare)
{
	int i;

	dump_ind();
	printf("PREPARE %s\n", prepare->name);
	dump_indent += 2;
	if (prepare->nparams)
	{
		dump_ind();
		printf("    Params (typeid): ");
		for(i = 0; i< prepare->nparams; i++)
			printf("%u ", prepare->params[i]);
		printf("\n");
	}
	dump_ind();
	printf("    STATEMENT ");
	dump_expr(prepare->expr);
	printf("\n");
	dump_indent -= 2;
}

static void
dump_execute(PLpgPSM_stmt_execute *stmt)
{
	dump_ind();
	
	if (!stmt->name)
	{
		printf("EXECUTE IMMEDIATE ");
		dump_expr(stmt->query);
		printf("\n");
	}
	else
	{
		printf("EXECUTE %s", stmt->name);
		printf("\n");
	}

	dump_indent += 2;
	if (stmt->rec != NULL)
	{
		dump_ind();
		printf("    INTO%s target = %d %s\n",
			   stmt->strict ? " STRICT" : "",
			   stmt->rec->dno, stmt->rec->refname);
	}
	if (stmt->row != NULL)
	{
		dump_ind();
		printf("    INTO%s target = %d %s\n",
			   stmt->strict ? " STRICT" : "",
			   stmt->row->dno, stmt->row->refname);
	}

	if (stmt->query)
	{
		int i;

		dump_ind();
		printf("    USING {");
		
		for (i = 0; i < stmt->query->nparams; i++)
		{
			if (i > 0) 
				printf(", ");
			printf("$%d=%d", i + 1, stmt->query->params[i]);
		}
		printf("}\n");		
    	}

	dump_indent -= 2;
}


static void
dump_getdiag(PLpgPSM_stmt_getdiag *stmt)
{
	ListCell   *lc;

	dump_ind();
	printf("GET DIAGNOSTICS ");
	foreach(lc, stmt->diag_items)
	{
		PLpgPSM_diag_item *diag_item = (PLpgPSM_diag_item *) lfirst(lc);

		if (lc != list_head(stmt->diag_items))
			printf(", ");

		printf("{var %d} = ", diag_item->target);

		switch (diag_item->kind)
		{
			case PLPGPSM_GETDIAG_ROW_COUNT:
				printf("ROW_COUNT");
				break;

			case PLPGPSM_GETDIAG_RESULT_OID:
				printf("RESULT_OID");
				break;

			default:
				printf("???");
				break;
		}
	}
	printf("\n");
}

static void
dump_expr(PLpgPSM_expr *expr)
{
	int			i;

	printf("'%s", expr->query);
	if (expr->nparams > 0)
	{
		printf(" {");
		for (i = 0; i < expr->nparams; i++)
		{
			if (i > 0)
				printf(", ");
			printf("$%d=%d", i + 1, expr->params[i]);
		}
		printf("}");
	}
	printf("'");
}

void
plpgpsm_dumptree(PLpgPSM_function *func)
{
	int			i;
	PLpgPSM_datum *d;

	printf("\nExecution tree of successfully compiled PL/pgSQL function %s:\n",
		   func->fn_name);

	printf("\nFunction's data area:\n");
	for (i = 0; i < func->ndatums; i++)
	{
		d = func->datums[i];

		printf("    entry %d: ", i);
		switch (d->dtype)
		{
			case PLPGPSM_DTYPE_VAR:
				{
					PLpgPSM_var *var = (PLpgPSM_var *) d;

					printf("VAR %-16s type %s (typoid %u) atttypmod %d\n",
						   var->refname, var->datatype->typname,
						   var->datatype->typoid,
						   var->datatype->atttypmod);
					if (var->isconst)
						printf("                                  CONSTANT\n");
					if (var->notnull)
						printf("                                  NOT NULL\n");
					if (var->default_val != NULL)
					{
						printf("                                  DEFAULT ");
						dump_expr(var->default_val);
						printf("\n");
					}
					if (var->cursor_explicit_expr != NULL)
					{
						if (var->cursor_explicit_argrow >= 0)
							printf("                                  CURSOR argument row %d\n", var->cursor_explicit_argrow);

						printf("                                  CURSOR IS ");
						dump_expr(var->cursor_explicit_expr);
						printf("\n");
					}
					if (var->cursor_options & CURSOR_OPT_SCROLL)
						printf("                                  SCROLLABLE ");
					if (var->prepname)
						printf("                                  DYNAMIC CURSOR FOR PREPARED STATEMENT NAME %s", var->prepname);
				}
				break;
			case PLPGPSM_DTYPE_ROW:
				{
					PLpgPSM_row *row = (PLpgPSM_row *) d;
					int			i;

					printf("ROW %-16s fields", row->refname);
					for (i = 0; i < row->nfields; i++)
					{
						if (row->fieldnames[i])
							printf(" %s=var %d", row->fieldnames[i],
								   row->varnos[i]);
					}
					printf("\n");
				}
				break;
			case PLPGPSM_DTYPE_REC:
				printf("REC %s\n", ((PLpgPSM_rec *) d)->refname);
				break;
			case PLPGPSM_DTYPE_RECFIELD:
				printf("RECFIELD %-16s of REC %d\n",
					   ((PLpgPSM_recfield *) d)->fieldname,
					   ((PLpgPSM_recfield *) d)->recparentno);
				break;
			case PLPGPSM_DTYPE_ARRAYELEM:
				printf("ARRAYELEM of VAR %d subscript ",
					   ((PLpgPSM_arrayelem *) d)->arrayparentno);
				dump_expr(((PLpgPSM_arrayelem *) d)->subscript);
				printf("\n");
				break;
			case PLPGPSM_DTYPE_TRIGARG:
				printf("TRIGARG ");
				dump_expr(((PLpgPSM_trigarg *) d)->argnum);
				printf("\n");
				break;
			default:
				printf("??? unknown data type %d\n", d->dtype);
		}
	}
	printf("\nFunction's statements:\n");

	dump_indent = 1;
	dump_stmt(func->action);
	printf("\nEnd of execution tree of function %s\n\n", func->fn_name);
	fflush(stdout);
}


static void
dump_cursor_direction(PLpgPSM_stmt_fetch *stmt)
{
	dump_indent += 2;
	dump_ind();
	switch (stmt->direction)
	{
		case FETCH_FORWARD:
			printf("    FORWARD ");
			break;
		case FETCH_BACKWARD:
			printf("    BACKWARD ");
			break;
		case FETCH_ABSOLUTE:
			printf("    ABSOLUTE ");
			break;
		case FETCH_RELATIVE:
			printf("    RELATIVE ");
			break;
		default:
			printf("??? unknown cursor direction %d\n", stmt->direction);
	}
	
	if (stmt->expr)
	{
		dump_expr(stmt->expr);
		printf("\n");
	}
	else
		printf("%d\n", stmt->how_many);
		
	dump_indent -= 2;
}
