/*
 * pg_filesystem.c
 */

#include "postgres.h"
#include "funcapi.h"
#include "utils/builtins.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>

#ifdef _WIN32
#else
#include <sys/vfs.h>
#include <blkid/blkid.h>
#endif


extern Datum pg_fs_size(PG_FUNCTION_ARGS);
extern Datum pg_fs_avail(PG_FUNCTION_ARGS);
extern Datum pg_statfs(PG_FUNCTION_ARGS);
extern Datum pg_fs_devid(PG_FUNCTION_ARGS);
extern Datum pg_fs_devname(PG_FUNCTION_ARGS);

PG_FUNCTION_INFO_V1(pg_fs_size);
PG_FUNCTION_INFO_V1(pg_fs_avail);
PG_FUNCTION_INFO_V1(pg_statfs);
PG_FUNCTION_INFO_V1(pg_fs_devid);
PG_FUNCTION_INFO_V1(pg_fs_devname);

#define PARTICULAR_DEVICE  "non_physical_device"

#if PG_VERSION_NUM < 80400

#define TextDatumGetCString

#define text_to_cstring(textp) \
	DatumGetCString(DirectFunctionCall1(textout, PointerGetDatum(textp)))

static text *
cstring_to_text(char *str)
{
	int			len = strlen(str) ;
	text	   *result = (text *) palloc(len + VARHDRSZ);

	SET_VARSIZE(result, len + VARHDRSZ);
	memcpy(VARDATA(result), str, len);

	return result;
}

#endif

#ifdef _WIN32
static bool
fs_size(const char *path, int64 *totalBytes, int64 *availBytes)
{
	ULARGE_INTEGER		avail;
	ULARGE_INTEGER		size;

	if (!GetDiskFreeSpaceEx(path, &avail, &size, NULL))
		return false;

	if (totalBytes)
		*totalBytes = size.QuadPart;
	if (availBytes)
		*availBytes = avail.QuadPart;
	return true;
}
#else

static bool
fs_size(const char *path, int64 *totalBytes, int64 *availBytes)
{
	struct statfs sf;

	if (statfs(path, &sf) == -1)
		return false;

	if (totalBytes)
		*totalBytes = (int64) sf.f_bsize * (int64) sf.f_blocks;
	if (availBytes)
		*availBytes = (int64) sf.f_bsize * (int64) sf.f_bfree;
	return true;
}
#endif


Datum
pg_fs_size(PG_FUNCTION_ARGS)
{
    const char	   *path = text_to_cstring(PG_GETARG_TEXT_P(0));
	int64			size;

	if (!fs_size(path, &size, NULL))
		ereport(ERROR,
				(errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
				 errmsg("faild statfs on %s", path)));

	PG_RETURN_INT64(size);
}

	
Datum
pg_fs_avail(PG_FUNCTION_ARGS)
{
    const char	   *path = text_to_cstring(PG_GETARG_TEXT_P(0));
	int64			avail;

	if (!fs_size(path, NULL, &avail))
		ereport(ERROR,
				(errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
				 errmsg("faild statfs on %s", path)));

	PG_RETURN_INT64(avail);
}

Datum
pg_fs_devid(PG_FUNCTION_ARGS)
{
	long devid = 0;
	struct stat st;
    const char  *path = text_to_cstring(PG_GETARG_TEXT_P(0));

	if (stat(path, &st) == -1)
	{
        ereport(ERROR,
                (errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
                 errmsg("faild stat on %s", path)));
	}

	devid = st.st_dev;

	PG_RETURN_INT64(devid);
}

Datum
pg_fs_devname(PG_FUNCTION_ARGS)
{
#ifdef _WIN32
	char	dev_path[MAX_PATH+1];
	//char	devname[MAX_PATH+1];
	//char	fsname[MAX_PATH+1];
	char   *path = text_to_cstring(PG_GETARG_TEXT_P(0));


	/* Get Mount path */
	if(!GetVolumePathName(
		path, dev_path, lengthof(dev_path)))
    {
        ereport(ERROR,
                (errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
                 errmsg("faild get volume path on %s", path)));
    }


	/* Under Construction... */
	/*
	if (!GetVolumeInformation(
		dev_path,
		devname, lengthof(devname),
		NULL, NULL, NULL,
		NULL, 0))
    {
        ereport(ERROR,
                (errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
                 errmsg("faild get volume name on %s", path)));
    }
	*/
	PG_RETURN_TEXT_P(cstring_to_text(dev_path));

#else
	struct stat st;
	const char *path = text_to_cstring(PG_GETARG_TEXT_P(0));
	char *devname;
	char *non_devname = PARTICULAR_DEVICE;

    if (stat(path, &st) == -1)
    {
        ereport(ERROR,
                (errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
                 errmsg("faild stat on %s", path)));
    }

	devname = blkid_devno_to_devname(st.st_dev);
	if (devname == NULL)
	{
		PG_RETURN_TEXT_P(cstring_to_text(non_devname));
	}

	PG_RETURN_TEXT_P(cstring_to_text(devname));
#endif
}

