/*--------------------------------------------------------------------
 * FILE:
 *    set_key.c
 *
 * NOTE:
 *
 * Portions Copyright (c) 2003-2009, Atsushi Mitani
 *--------------------------------------------------------------------
 */
#include "postgres.h"

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <fcntl.h>
#include <ctype.h>
#include <sys/stat.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/sem.h>
#include <netdb.h>
#include <errno.h>
#include <sys/file.h>

#include "libpq-fe.h"
#include "libpq-int.h"
#include "fe-auth.h"

#include "replicate.h"
#include "libpgc/libpgc.h"
#include "pglb/pglb.h"
#include "pgrp/pgreplicate.h"
#include "admin/pgc_admin.h"

int PGRcreate_public_key_files(char * path);
int PGRsend_public_key_files(char * path);
int PGRcreate_admin_pem_files(char * path);
int PGRcreate_probe_pem_files(char * path);

static int get_each_public_key(char * dest, Physical_Server_Info * server);
static int add_user_public_key_to_autholized_keys_file(char * w_path, char * r_path, int add_flag);
static int add_host_public_key_to_known_hosts_file(char * w_path, char * r_path, Physical_Server_Info * server, int add_flag);
static FILE * open_file(char * path, char * fname, char * mode);
static int set_file_mode(char * path, char * fname, mode_t mode);
static int send_public_key_file(char * r_path, Physical_Server_Info * server);
static int modify_ca_conf(char * conf_dir, char * workPath, char * fname, char * hostName);
static int create_each_server_pem(char * r_path, char * conf_dir, Physical_Server_Info * pserver, SSL_Server_Info * server, char * conf);
static int exec_cp(char * src_path, char * src_fname, char * dest_path, char * dest_fname);
static int exec_server_pem(char * bin_path, char * w_path, char * conf);
static int create_each_client_pem(char * r_path, char * conf_dir, Physical_Server_Info * server, SSL_Server_Info * client, char * conf);
static int append_to_file(char * dest, char * src);
static int exec_client_pem(char * bin_path, char * w_path, char * conf);

int
PGRcreate_public_key_files(char * path)
{

	char * func = "create_public_key_files()";
	char * buf = NULL;
	int buf_size = 512;
	char * conf_dir;
	int i = 0;
	char * p = NULL;
	Physical_Server_Info * server = PhysicalServerTbl;

	if (path == NULL) 
	{
		return STATUS_ERROR;
	}
	
	if ((buf = malloc( RecordNumTbl->serverNum * buf_size)) == NULL)
	{
		show_error("%s: malloc failed(%s)",func,strerror(errno));
		return STATUS_ERROR;
	}
	memset(buf, 0, RecordNumTbl->serverNum * buf_size);
	/* get public key files from each servers */
	conf_dir = buf;
	for (i = 0 ; i < RecordNumTbl->serverNum ; i ++)
	{
		if ((server+i) == NULL)
		{
			show_error("%s: PGRget_Physical_Server_Rec failed",func);
			continue;
		}
		if ((p = strchr((server+i)->hostName,'.')) != NULL)
		{
			*p = '\0';
		}
		sprintf(conf_dir,"%s/%s",path,(server+i)->hostName);
		if (p != NULL)
			*p = '.';
		get_each_public_key(conf_dir, (server+i));
		conf_dir += buf_size;
	}

	/* create authorized_keys file & known_hosts file */
	conf_dir = buf;
	for (i = 0 ; i < RecordNumTbl->serverNum ; i ++)
	{
		add_user_public_key_to_autholized_keys_file(path, conf_dir, i);
		add_host_public_key_to_known_hosts_file(path, conf_dir,(server+i), i);
		conf_dir += buf_size;
	}

	free(buf);
	return STATUS_OK;
}

static int
get_each_public_key(char * dest, Physical_Server_Info * server)
{
	char * func = "get_each_public_key()";
	char scp_command[1024];
	memset(scp_command, 0, sizeof(scp_command));

	if ((dest == NULL) || (server == NULL))
	{
		show_error("%s:args is NULL",func);
		return STATUS_ERROR;
	}
	sprintf(scp_command,"scp -p %s@%s:~%s/.ssh/%s %s@%s:/etc/ssh/%s %s",server->userName, server->hostName, server->userName, USER_RSA_PUB_KEY, server->userName,server->hostName, HOST_RSA_PUB_KEY, dest);

	system(scp_command);
	return STATUS_OK;
}

static int
add_user_public_key_to_autholized_keys_file(char * w_path, char * r_path, int add_flag)
{
	char * func = "add_user_public_key_to_autholized_keys_file()";
	FILE * rfp = NULL;
	FILE * wfp = NULL;
	char * buf = NULL;
	int size = 0;
	char * mode = NULL;

	if ((w_path == NULL) || (r_path == NULL))
	{
		return STATUS_ERROR;
	}

	/* read file open */
	if ((rfp = open_file(r_path, USER_RSA_PUB_KEY, "r")) == NULL)
	{
		show_error("%s: %s file open failed(%s)",func,USER_RSA_PUB_KEY,strerror(errno));
		return STATUS_ERROR;
	}

	mode = (add_flag == 0)?"w":"a";
	/* write file open */
	if ((wfp = open_file(w_path, AUTHORIZED_KEYS_FILE, mode)) == NULL)
	{
		show_error("%s: %s file open failed(%s)",func,AUTHORIZED_KEYS_FILE, strerror(errno));
		return STATUS_ERROR;
	}

	size = 2048;
	if ((buf = malloc(size)) == NULL)
	{
		show_error("%s: buf malloc failed(%s)",func,strerror(errno));
		return STATUS_ERROR;
	}
	memset(buf, 0, size);

	while (fgets(buf, size, rfp) != NULL)
	{
		fputs(buf,wfp);
	}
	fclose(rfp);
	fflush(wfp);
	fclose(wfp);
	free(buf);
	set_file_mode(w_path, AUTHORIZED_KEYS_FILE, S_IRUSR | S_IWUSR);

	return STATUS_OK;
}

static int
add_host_public_key_to_known_hosts_file(char * w_path, char * r_path, Physical_Server_Info * server, int add_flag)
{

	char * func = "add_host_public_key_to_known_hosts_file()";
	FILE * rfp = NULL;
	FILE * wfp = NULL;
	char * buf = NULL;
	char * p = NULL;
	char * mode = NULL;
	int size = 0;

	if ((w_path == NULL) || (r_path == NULL))
	{
		return STATUS_ERROR;
	}

	/* read file open */
	if ((rfp = open_file(r_path, HOST_RSA_PUB_KEY, "r")) == NULL)
	{
		show_error("%s: %s file open failed(%s)",func,HOST_RSA_PUB_KEY, strerror(errno));
		return STATUS_ERROR;
	}

	mode = (add_flag == 0)?"w":"a";
	/* write file open */
	if ((wfp = open_file(w_path, KNOWN_HOSTS_FILE, mode)) == NULL)
	{
		show_error("%s: %s file open failed(%s)",func, KNOWN_HOSTS_FILE, strerror(errno));
		return STATUS_ERROR;
	}

	size = 2048;
	if ((buf = malloc(size)) == NULL)
	{
		show_error("%s: buf malloc failed(%s)",func,strerror(errno));
		return STATUS_ERROR;
	}
	memset(buf, 0, size);

	sprintf(buf,"%s,%s ",server->hostName, server->ipAddr);
	p = strchr(buf,'\0');
	size -= strlen(buf);

	while (fgets(p, size, rfp) != NULL)
	{
		fputs(buf,wfp);
	}
	fclose(rfp);
	fflush(wfp);
	fclose(wfp);
	free(buf);
	set_file_mode(w_path, KNOWN_HOSTS_FILE, S_IRUSR | S_IWUSR);

	return STATUS_OK;
}

static FILE *
open_file(char * path, char * fname, char * mode)
{
	char * func = "open_file";
	int size = 0;
	char * fname_with_path = NULL;
	FILE * fp = NULL;

	if ((path == NULL) || (fname == NULL))
	{
		return NULL;
	}

	size = strlen(path) + strlen(fname) + 4;
	if ((fname_with_path = malloc(size)) == NULL)
	{
		show_error("%s: fname_with_path malloc failed(%s)",func,strerror(errno));
		return NULL;
	}
	memset(fname_with_path, 0, size);
	sprintf(fname_with_path,"%s/%s",path, fname);

	/* read file open */
	if ((fp = fopen(fname_with_path, mode)) == NULL)
	{
		show_error("%s: %s file open failed(%s)",func,fname_with_path,strerror(errno));
		free(fname_with_path);
		return NULL;
	}
	free(fname_with_path);

	return fp;
}

static int
set_file_mode(char * path, char * fname, mode_t mode)
{
	char * func = "set_file_mode()";
	int size = 0;
	char * fname_with_path = NULL;
	int rtn ;

	if ((path == NULL) || (fname == NULL))
	{
		return STATUS_ERROR;
	}

	size = strlen(path) + strlen(fname) + 4;
	if ((fname_with_path = malloc(size)) == NULL)
	{
		show_error("%s: fname_with_path malloc failed(%s)",func,strerror(errno));
		return STATUS_ERROR;
	}
	memset(fname_with_path, 0, size);
	sprintf(fname_with_path,"%s/%s",path, fname);

	rtn = chmod(fname_with_path,mode);

	free(fname_with_path);

	return rtn;
}

int
PGRsend_public_key_files(char * path)
{
	Physical_Server_Info * server = PhysicalServerTbl;
	int i;

	/* send authorized_keys file & known_hosts file to each server */
	for (i = 0 ; i < RecordNumTbl->serverNum ; i ++)
	{
		send_public_key_file(path,(server+i));
	}


	return STATUS_OK;
}

static int
send_public_key_file(char * r_path, Physical_Server_Info * server)
{
	char * func="send_public_key_file()";
	char scp_command[1024];

	if ((r_path == NULL) || (server == NULL))
	{
		show_error("%s:args is NULL",func);
		return STATUS_ERROR;
	}

	memset(scp_command, 0, sizeof(scp_command));

	sprintf(scp_command,"scp -p %s/%s %s/%s %s@%s:~%s/.ssh",r_path, AUTHORIZED_KEYS_FILE, r_path, KNOWN_HOSTS_FILE, server->userName, server->hostName, server->userName);

	system(scp_command);
	return STATUS_OK;
}

int
PGRcreate_admin_pem_files(char * path)
{

	char * func = "PGRcreate_admin_pem_files()";
	char conf_dir[512];
	int i = 0;
	Physical_Server_Info * server = NULL;
	SSL_Server_Info * admin = AdminTbl;

	if (path == NULL) 
	{
		show_error("%s:args is NULL",func);
		return STATUS_ERROR;
	}
	
	for (i = 0 ; i < RecordNumTbl->adminNum ; i ++, admin ++)
	{
		memset(conf_dir, 0, sizeof(conf_dir));
		server = PGRget_Physical_Server_Rec(admin->physicalServerId);
		if (PGRget_dir_name_with_hostname(conf_dir, path, server) == NULL)
		{
			continue;
		}
		create_each_server_pem(path, conf_dir, server, admin, ADMIN_CNF_FILE);
		create_each_client_pem(path, conf_dir, server, admin, ADMIN_CNF_FILE);
	}
	return STATUS_OK;
}

static int
modify_ca_conf(char * conf_dir, char * workPath, char * fname, char * hostName)
{
	char * func ="modify_ca_conf()";
	FILE * fp = NULL;
	FILE * tmpfp = NULL;
	char buf[256];
	char * tmp_fname = "/tmp/modify_ca_conf.tmp";
	/* update admin.cnf file */
	if ((fp = PGRopen_write_file(conf_dir, workPath, fname, "r+")) == NULL)
	{
		show_error("%s: %s file open error",func, fname);
		return STATUS_ERROR;
	}
	if ((tmpfp = fopen(tmp_fname, "w")) == NULL)
	{
		show_error("%s:(%s)PGRopen_write_file failed",func,tmp_fname);
		return STATUS_ERROR;
	}

	sprintf(buf,"DNS:%s,DNS:localhost",hostName);
	fp = PGRupdate_conf_contents(fp,tmpfp,"subjectAltName = ", buf);
	sprintf(buf,"%s",hostName);
	fp = PGRupdate_conf_contents(fp,tmpfp,"commonName             = ", buf);
	sprintf(buf,"DNS:%s,DNS:localhost",hostName);
	fp = PGRupdate_conf_contents(fp,tmpfp,"subjectAltName = ", buf);
	tmpfp = PGRcopy_file(fp,tmpfp);
	fclose(fp);
	fclose(tmpfp);

	if ((fp = PGRopen_write_file(conf_dir, workPath, fname, "w")) == NULL)
	{
		show_error("%s:(%s)PGRopen_write_file failed",func,fname);
		return STATUS_ERROR;
	}
	if ((tmpfp = fopen(tmp_fname, "r")) == NULL)
	{
		show_error("%s:(%s)open failed",func,fname);
		return STATUS_ERROR;
	}
	fp = PGRcopy_file(tmpfp,fp);
	fclose(tmpfp);
	fclose(fp);
	unlink(tmp_fname);

	return STATUS_OK;
}

static int
create_each_server_pem(char * r_path, char * conf_dir, Physical_Server_Info * pserver, SSL_Server_Info * server, char * conf)
{
	char * func = "create_each_server_pem()";
	char w_path[512];
	char *p = NULL;
	char buf[64];

	show_debug("%s:workPath[%s]",func,server->workPath);
	p = strrchr(server->workPath, '/');
	if (p == NULL)
	{
		show_error("%s:work path is not set",func);
		return STATUS_ERROR;
	}
	sprintf(w_path,"%s/%s",conf_dir, ++p);
	/* copy common pem files */
	exec_cp(r_path, "dh512.pem", w_path, "dh512.pem");
	exec_cp(r_path, "dh1024.pem", w_path, "dh1024.pem");
	exec_cp(r_path, "rootcert.pem", w_path, "rootcert.pem");
	exec_cp(r_path, "root.pem", w_path, "root.pem");
	exec_cp(r_path, "serverCA.cnf.sample", w_path, "serverCA.cnf");
	/*
	exec_cp(r_path, "serverCA.pem", w_path, "serverCA.pem");
	*/
	/* copy admin.cnf file */
	sprintf(buf,"%s.sample",conf);
	exec_cp(r_path, buf, w_path, conf);
	modify_ca_conf(conf_dir, server->workPath, conf, server->hostName);
	return (exec_server_pem(server->sslPath, w_path, conf));
}

static int
exec_cp(char * src_path, char * src_fname, char * dest_path, char * dest_fname)
{
	char src[256];
	char dest[256];
	int i = 0;
	char * args[8];
	int rtn;

	sprintf(src,"%s/%s",src_path, src_fname);
	sprintf(dest,"%s/%s",dest_path, dest_fname);
	i = 0;
	args[i++] = "cp";
	args[i++] = src;
	args[i++] = dest;
	args[i++] = NULL;
	rtn = PGR_execv("/bin/cp", args);	

	return rtn;
}
static int
exec_server_pem(char * bin_path, char * w_path, char * conf)
{
	int rtn = STATUS_OK;
	char * args[24];
	int i = 0;
	char command[128];
	char key_pem[256];
	char req_pem[256];
	char cert_pem[256];
	char server_pem[256];
	char server_cnf[256];
	char server_ca[256];
	char server_ca_cnf[256];
	char server_ca_key[256];
	char server_ca_req[256];
	char server_ca_cert[256];
	char root_cert[256];
	char root_ca[256];

	sprintf(command,"%s/openssl",bin_path);
	sprintf(key_pem,"%s/serverkey.pem",w_path);
	sprintf(req_pem,"%s/serverreq.pem",w_path);
	sprintf(cert_pem,"%s/servercert.pem",w_path);
	sprintf(server_cnf,"%s/%s",w_path,conf);
	sprintf(server_ca,"%s/serverCA.pem",w_path);
	sprintf(server_ca_cnf,"%s/serverCA.cnf",w_path);
	sprintf(server_ca_key,"%s/serverCAkey.pem",w_path);
	sprintf(server_ca_req,"%s/serverCAreq.pem",w_path);
	sprintf(server_ca_cert,"%s/serverCAcert.pem",w_path);
	sprintf(root_cert,"%s/rootcert.pem",w_path);
	sprintf(root_ca,"%s/root.pem",w_path);
	sprintf(server_pem,"%s/server.pem",w_path);

	/* create serverCAkey.pem and serverCAreq.pem */
	i = 0;
	args[i++] = "openssl";
	args[i++] = "req";
	args[i++] = "-nodes";
	args[i++] = "-newkey";
	args[i++] = "rsa:1024";
	args[i++] = "-sha1";
	args[i++] = "-keyout";
	args[i++] = server_ca_key;
	args[i++] = "-out";
	args[i++] = server_ca_req;
	args[i++] = "-config";
	args[i++] = server_ca_cnf;
	args[i++] = NULL;

	rtn = PGR_execv(command, args);	
	/* create serverCAcert.pem */
	i = 0;
	args[i++] = "openssl";
	args[i++] = "x509";
	args[i++] = "-req";
	args[i++] = "-in";
	args[i++] = server_ca_req;
	args[i++] = "-sha1";
	args[i++] = "-extfile";
	args[i++] = server_ca_cnf;
	args[i++] = "-extensions";
	args[i++] = "certificate_extensions";
	args[i++] = "-CA";
	args[i++] = root_ca;
	args[i++] = "-CAkey";
	args[i++] = root_ca;
	args[i++] = "-CAcreateserial";
	args[i++] = "-out";
	args[i++] = server_ca_cert;
	args[i++] = NULL;

	rtn = PGR_execv(command, args);	

	/* create serverCA.pem */
	i = 0;
	args[i++] = "cp";
	args[i++] = server_ca_cert;
	args[i++] = server_ca;
	args[i++] = NULL;
	rtn = PGR_execv("/bin/cp", args);	
	append_to_file(server_ca, server_ca_key);
	append_to_file(server_ca, root_cert);

	/* create serverkey.pem and serverreq.pem */
	i = 0;
	args[i++] = "openssl";
	args[i++] = "req";
	args[i++] = "-nodes";
	args[i++] = "-newkey";
	args[i++] = "rsa:1024";
	args[i++] = "-sha1";
	args[i++] = "-keyout";
	args[i++] = key_pem;
	args[i++] = "-out";
	args[i++] = req_pem;
	args[i++] = "-config";
	args[i++] = server_cnf;
	args[i++] = "-reqexts";
	args[i++] = "req_extensions";
	args[i++] = NULL;

	rtn = PGR_execv(command, args);	
	
	/* create servercert.pem */
	i = 0;
	args[i++] = "openssl";
	args[i++] = "x509";
	args[i++] = "-req";
	args[i++] = "-in";
	args[i++] = req_pem;
	args[i++] = "-sha1";
	args[i++] = "-extfile";
	args[i++] = server_cnf;
	args[i++] = "-extensions";
	args[i++] = "certificate_extensions";
	args[i++] = "-CA";
	args[i++] = server_ca;
	args[i++] = "-CAkey";
	args[i++] = server_ca;
	args[i++] = "-CAcreateserial";
	args[i++] = "-out";
	args[i++] = cert_pem;
	args[i++] = NULL;

	rtn = PGR_execv(command, args);	

	/* create server.pem */
	i = 0;
	args[i++] = "cp";
	args[i++] = cert_pem;
	args[i++] = server_pem;
	args[i++] = NULL;
	rtn = PGR_execv("/bin/cp", args);	
	append_to_file(server_pem, key_pem);
	append_to_file(server_pem, server_ca_cert);
	append_to_file(server_pem, root_cert);

	return rtn;
}

static int
append_to_file(char * dest, char * src)
{
	FILE * rfp = NULL;
	FILE * wfp = NULL;
	char buf[512];

	if ((wfp = fopen(dest , "a")) == NULL)
	{
		show_error("%s can not open",dest);
		return STATUS_ERROR;
	}
	if ((rfp = fopen(src, "r")) == NULL)
	{
		show_error("%s can not open",dest);
		return STATUS_ERROR;
	}
	while (fgets(buf, sizeof(buf), rfp) != NULL)
	{
		fputs(buf, wfp);
	}
	fflush(wfp);
	fclose(wfp);
	fclose(rfp);

	return STATUS_OK;
}

int
PGRcreate_probe_pem_files(char * path)
{

	char * func = "PGRcreate_probe_pem_files()";
	char conf_dir[512];
	int i = 0;
	Physical_Server_Info * server = NULL;
	SSL_Server_Info * probe = ProbeTbl;

	if (path == NULL) 
	{
		show_error("%s:args is NULL",func);
		return STATUS_ERROR;
	}
	
	for (i = 0 ; i < RecordNumTbl->probeNum ; i ++, probe ++)
	{
		memset(conf_dir, 0, sizeof(conf_dir));
		server = PGRget_Physical_Server_Rec(probe->physicalServerId);
		if (PGRget_dir_name_with_hostname(conf_dir, path, server) == NULL)
		{
			continue;
		}
		create_each_server_pem(path, conf_dir, server, probe, PROBE_CNF_FILE );
		create_each_client_pem(path, conf_dir, server, probe, PROBE_CNF_FILE );
	}
	return STATUS_OK;
}

static int
create_each_client_pem(char * r_path, char * conf_dir, Physical_Server_Info * server, SSL_Server_Info * client, char * conf)
{
	char * func ="create_each_client_pem()";
	char w_path[512];
	char *p = NULL;

	p = strrchr(client->workPath, '/');
	if (p == NULL)
	{
		show_error("%s:work path is not set",func);
		return STATUS_ERROR;
	}
	sprintf(w_path,"%s/%s",conf_dir, ++p);
	/* copy common pem files */
	exec_cp(r_path, "dh512.pem", w_path, "dh512.pem");
	exec_cp(r_path, "dh1024.pem", w_path, "dh1024.pem");
	exec_cp(r_path, "root.pem", w_path, "root.pem");
	exec_cp(r_path, "rootcert.pem", w_path, "rootcert.pem");
	return (exec_client_pem(client->sslPath, w_path, conf));
}

static int
exec_client_pem(char * bin_path, char * w_path, char * conf)
{
	int rtn = STATUS_OK;
	char * args[24];
	int i = 0;
	char command[128];
	char key_pem[256];
	char req_pem[256];
	char cert_pem[256];
	char client_pem[256];
	char client_cnf[256];
	char root_cert[256];
	char root_pem[256];

	sprintf(command,"%s/openssl",bin_path);
	sprintf(key_pem,"%s/clientkey.pem",w_path);
	sprintf(req_pem,"%s/clientreq.pem",w_path);
	sprintf(cert_pem,"%s/clientcert.pem",w_path);
	sprintf(client_cnf,"%s/%s",w_path,conf);
	sprintf(root_cert,"%s/rootcert.pem",w_path);
	sprintf(root_pem,"%s/root.pem",w_path);
	sprintf(client_pem,"%s/client.pem",w_path);

	/* create probekey.pem and probereq.pem */
	i = 0;
	args[i++] = "openssl";
	args[i++] = "req";
	args[i++] = "-nodes";
	args[i++] = "-newkey";
	args[i++] = "rsa:1024";
	args[i++] = "-sha1";
	args[i++] = "-keyout";
	args[i++] = key_pem;
	args[i++] = "-out";
	args[i++] = req_pem;
	args[i++] = "-config";
	args[i++] = client_cnf;
	args[i++] = "-reqexts";
	args[i++] = "req_extensions";
	args[i++] = NULL;

	rtn = PGR_execv(command, args);	
	
	/* create probecert.pem */
	i = 0;
	args[i++] = "openssl";
	args[i++] = "x509";
	args[i++] = "-req";
	args[i++] = "-in";
	args[i++] = req_pem;
	args[i++] = "-sha1";
	args[i++] = "-extfile";
	args[i++] = client_cnf;
	args[i++] = "-extensions";
	args[i++] = "certificate_extensions";
	args[i++] = "-CA";
	args[i++] = root_pem;
	args[i++] = "-CAkey";
	args[i++] = root_pem;
	args[i++] = "-CAcreateserial";
	args[i++] = "-out";
	args[i++] = cert_pem;
	args[i++] = NULL;

	rtn = PGR_execv(command, args);	

	/* create client.pem */
	i = 0;
	args[i++] = "cp";
	args[i++] = cert_pem;
	args[i++] = client_pem;
	args[i++] = NULL;
	rtn = PGR_execv("/bin/cp", args);	
	append_to_file(client_pem, key_pem);
	append_to_file(client_pem, root_cert);

	return rtn;
}
