/*--------------------------------------------------------------------
 * FILE:
 *     cascade.c
 *
 * NOTE:
 *     This file is composed of the functions to call with the source
 *     at pgreplicate for backup and cascade .
 *
 * Portions Copyright (c) 2003, Atsushi Mitani
 *--------------------------------------------------------------------
 */
#ifdef USE_REPLICATION

#include "postgres.h"
#include "postgres_fe.h"

#include <stdio.h>
#include <unistd.h>
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif
#include <errno.h>
#include <ctype.h>
#include <time.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/sem.h>
#include <signal.h>
#include <sys/socket.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <netdb.h>
#ifdef HAVE_NETINET_TCP_H
#include <netinet/tcp.h>
#endif
#include <dirent.h>
#include <arpa/inet.h>

#ifdef HAVE_CRYPT_H
#include <crypt.h>
#endif

#ifdef MULTIBYTE
#include "mb/pg_wchar.h"
#endif

#include "libpq-fe.h"
#include "libpq-int.h"
#include "fe-auth.h"
#include "access/xact.h"
#include "replicate_com.h"
#include "pgreplicate.h"

int PGRstartup_cascade(void);
int PGRsend_lower_cascade(ReplicateHeader * header, char * query);
int PGRsend_upper_cascade(ReplicateHeader * header, char * query);
int PGRwait_answer_cascade(int  sock);
ReplicateServerInfo * PGRget_lower_cascade(void);
ReplicateServerInfo * PGRget_upper_cascade(void);
void PGRset_cascade_server_status(ReplicateServerInfo * cascade, int status);
ReplicateServerInfo * PGRrecv_cascade_answer(ReplicateServerInfo * cascade,ReplicateHeader * header);
int PGRsend_cascade(int sock , ReplicateHeader * header, char * query);
int PGRcascade_main(int sock, ReplicateHeader * header, char * query);
int PGRwait_notice_rlog_done(void);
int PGRsend_notice_rlog_done(int sock);

#if 0
static int count_cascade(int flag);
#endif
static ReplicateServerInfo * get_cascade_data(int * cnt, int flag);
static int add_cascade_data(ReplicateHeader * header, ReplicateServerInfo * add_data);
static int update_cascade_data(ReplicateHeader * header, ReplicateServerInfo * update_data);
static void write_cascade_status_file(ReplicateServerInfo * cascade);
static int notice_cascade_data(int sock);
static int notice_cascade_data_to_cluster_db(void);

/*--------------------------------------
 * PROTOTYPE DECLARATION
 *--------------------------------------
 */

#if 0
static int
count_cascade(int flag)
{
	int cnt = 0;
	int cascade_cnt = 0;
	ReplicateServerInfo * cascade = NULL;

	if ((Cascade_Tbl == NULL) || (Cascade_Inf == NULL))
	{
		return 0;
	}

	/* count cascadeing replication server */
	switch (flag)
	{
		case UPPER_CASCADE:
		case ALL_CASCADE:
			cascade = Cascade_Tbl;
			break;
		case LOWER_CASCADE:
			cascade = Cascade_Inf->myself;
			break;
	}

	if (cascade == NULL)
	{
		return 0;
	}
	while (cascade->useFlag != DATA_END)
	{
		if (cascade->useFlag == DATA_USE)
		{
			cascade_cnt ++;
		}
		if ((flag == UPPER_CASCADE) &&
			(cascade == Cascade_Inf->myself))
		{
			break;
		}
		cnt ++;
		if (cnt >= MAX_DB_SERVER -1 )
		{
			break;
		}
		cascade ++;
	}
	return cascade_cnt;
}
#endif /* if 0 */

static ReplicateServerInfo * 
get_cascade_data(int * cnt, int flag)
{
	char * func = "get_cascade_data()";
	int i = 0;
	int loop_cnt = 0;
	int size = 0;
	ReplicateServerInfo * buf = NULL;
	ReplicateServerInfo * cascade = NULL;

	size = sizeof(ReplicateServerInfo) * MAX_DB_SERVER;
	buf = (ReplicateServerInfo *)malloc(size);
	if (buf == (ReplicateServerInfo *)NULL)
	{
		show_error("%s:malloc failed: (%s)",func,strerror(errno));
		*cnt = 0;
		return NULL;
	}
	memset(buf,0,size);

	switch (flag)
	{
		case UPPER_CASCADE:
		case ALL_CASCADE:
			cascade = Cascade_Tbl;
			break;
		case LOWER_CASCADE:
			cascade = Cascade_Inf->myself;
			break;
	}

	if (cascade == NULL)
	{
		free(buf);
		*cnt = 0;
		return NULL;
	}
	i = 0;
	loop_cnt = 0;
	while (cascade->useFlag != DATA_END)
	{
		if (cascade->useFlag == DATA_USE) 
		{
			(buf + i)->useFlag = htonl(cascade->useFlag);
			strncpy((buf + i)->hostName,cascade->hostName,sizeof(cascade->hostName));
			(buf + i)->portNumber = htons(cascade->portNumber);
			(buf + i)->recoveryPortNumber = htons(cascade->recoveryPortNumber);
			(buf + i)->lifecheckPortNumber = htons(cascade->lifecheckPortNumber);
			i++;
		}
		if ((flag == UPPER_CASCADE) &&
			(cascade == Cascade_Inf->myself))
		{
			break;
		}
		loop_cnt ++;
		if (loop_cnt >= MAX_DB_SERVER -1 )
		{
			break;
		}
		if (Cascade_Inf->end == cascade)
		{
			break;
		}
		cascade ++;
	}
	*cnt = i;
	return buf;
}

static int
update_cascade_data(ReplicateHeader * header, ReplicateServerInfo * update_data)
{
	char * func = "update_cascade_data()";
	int size = 0;
	int cnt = 0;
	ReplicateServerInfo * ptr = NULL;
	ReplicateServerInfo * cascade = NULL;
	char hostName[HOSTNAME_MAX_LENGTH];

	if ((header == NULL ) || ( update_data == NULL))
	{
		show_error("%s:receive data is wrong",func);
		return STATUS_ERROR;
	}
	if ((Cascade_Tbl == NULL) || (Cascade_Inf == NULL))
	{
		show_error("%s:config data read error",func);
		return STATUS_ERROR;
	}
	size = ntohl(header->query_size);
	cnt = size / sizeof(ReplicateServerInfo);
	if (cnt >= MAX_DB_SERVER)
	{
		show_error("%s:update cascade data is too large. it's more than %d", func,MAX_DB_SERVER);
		return STATUS_ERROR;
	}

	Cascade_Inf->useFlag = DATA_INIT;
	if ((Cascade_Inf->upper != NULL) && (Cascade_Inf->upper->sock > 0))
	{
		close(Cascade_Inf->upper->sock);
	}
	if ((Cascade_Inf->lower != NULL) && (Cascade_Inf->lower->sock > 0))
	{
		close(Cascade_Inf->lower->sock);
	}
	Cascade_Inf->myself = NULL;
	Cascade_Inf->upper = NULL;
	Cascade_Inf->lower = NULL;

	gethostname(hostName,sizeof(hostName));
	ptr = update_data;
	cascade = Cascade_Tbl;
	memset(cascade,0,(sizeof(ReplicateServerInfo)*MAX_DB_SERVER));
	Cascade_Inf->top = cascade;
	while (cnt > 0)
	{
		cascade->useFlag = ntohl(ptr->useFlag);
		strncpy(cascade->hostName,ptr->hostName,sizeof(cascade->hostName));
		cascade->portNumber = ntohs(ptr->portNumber);
		cascade->recoveryPortNumber = ntohs(ptr->recoveryPortNumber);
		cascade->lifecheckPortNumber = ntohs(ptr->lifecheckPortNumber);

		if ((!strncmp(cascade->hostName,hostName,sizeof(cascade->hostName)))  &&
			(cascade->portNumber == Port_Number) &&
			(cascade->recoveryPortNumber == Recovery_Port_Number))
		{
			Cascade_Inf->myself = cascade;
		}
		Cascade_Inf->end = cascade;
		cascade ++;
		ptr ++;
		cnt --;
		cascade->useFlag = DATA_END;
	}
	Cascade_Inf->useFlag = DATA_USE;

	return STATUS_OK;
}

static int
add_cascade_data(ReplicateHeader * header, ReplicateServerInfo * add_data)
{
	char *func = "add_cascade_data()";
	int size = 0;
	int cnt = 0;
	ReplicateServerInfo * ptr = NULL;
	ReplicateServerInfo * cascade = NULL;
	char hostName[HOSTNAME_MAX_LENGTH];

	if ((header == NULL ) || ( add_data == NULL))
	{
		show_error("%s:receive data is wrong",func);
		return STATUS_ERROR;
	}
	if ((Cascade_Tbl == NULL) || (Cascade_Inf == NULL))
	{
		show_error("%s:config data read error",func);
		return STATUS_ERROR;
	}
	size = ntohl(header->query_size);
	cnt = size / sizeof(ReplicateServerInfo);
	if (cnt >= MAX_DB_SERVER)
	{
		show_error("%s:addtional cascade data is too large. it's more than %d", func,MAX_DB_SERVER);
		return STATUS_ERROR;
	}

	Cascade_Inf->useFlag = DATA_INIT;
	if ((Cascade_Inf->lower != NULL) && (Cascade_Inf->lower->sock > 0))
	{
		close(Cascade_Inf->lower->sock);
	}
	Cascade_Inf->lower = NULL;

	gethostname(hostName,sizeof(hostName));
	ptr = add_data;
	cascade = Cascade_Inf->myself;
	cascade ++;
	while (cnt > 0)
	{
		cascade->useFlag = ntohl(ptr->useFlag);
		strncpy(cascade->hostName,ptr->hostName,sizeof(cascade->hostName));
		cascade->portNumber = ntohs(ptr->portNumber);
		cascade->recoveryPortNumber = ntohs(ptr->recoveryPortNumber);
		cascade->lifecheckPortNumber = ntohs(ptr->lifecheckPortNumber);

		if ((!strncmp(cascade->hostName,hostName,sizeof(cascade->hostName)))  &&
			(cascade->portNumber == Port_Number) &&
			(cascade->recoveryPortNumber == Recovery_Port_Number))
		{
			ptr ++;
			cnt --;
			continue;
		}
		cascade ++;
		cascade->useFlag = DATA_END;
		ptr ++;
		cnt --;
	}
	Cascade_Inf->useFlag = DATA_USE;

	return STATUS_OK;
}

int
PGRstartup_cascade(void)
{
	char * func = "PGRstartup_cascade()";
	int cnt = 0;
	int status = STATUS_OK;
	ReplicateHeader header;
	ReplicateServerInfo * cascade = NULL;
	ReplicateServerInfo * buf = NULL;

	if ((Cascade_Tbl == NULL) || (Cascade_Inf == NULL))
	{
		show_error("%s:config data read error",func);
		return STATUS_ERROR;
	}

	/* count lower server */
	cascade = Cascade_Inf->myself;
	if (cascade == NULL)
	{
		show_error("%s:cascade data initialize error",func);
		return STATUS_ERROR;
	}
	buf = get_cascade_data(&cnt,LOWER_CASCADE);
	if (cnt <= 0)
	{
		show_error("%s:cascade data get error",func);
		return STATUS_ERROR;
	}
	
	memset(&header,0,sizeof(ReplicateHeader));
	header.cmdSys = CMD_SYS_CASCADE;
	header.cmdSts = CMD_STS_TO_UPPER;
	header.cmdType = CMD_TYPE_ADD;
	header.query_size = htonl(sizeof(ReplicateServerInfo) * cnt);

	status = PGRsend_upper_cascade(&header, (char *)buf);
	if (buf != NULL)
	{
		free(buf);
	}
	if (status == STATUS_OK)
	{
		memset(&header,0,sizeof(ReplicateHeader));
		buf = PGRrecv_cascade_answer( Cascade_Inf->upper, &header);
		if ((buf != NULL) &&
			(header.cmdSys == CMD_SYS_CASCADE) &&
			(header.cmdSts == CMD_STS_TO_LOWER) &&
			(header.cmdType == CMD_TYPE_UPDATE_ALL))
		{
			status = update_cascade_data(&header,buf);
			free(buf);
		}
	}
	
	return status;
}

int
PGRsend_lower_cascade(ReplicateHeader * header, char * query)
{
	char * func = "PGRsend_lower_cascade()";
	int status = STATUS_OK;

	for(;;)
	{
		if (Cascade_Inf->lower != NULL)
		{
			if (Cascade_Inf->lower->sock <= 0)
			{
#ifdef PRINT_DEBUG
				show_debug("%s:sock is not ready, Cascade_Inf->lower set NULL",func);
#endif			
				status = PGR_Create_Socket_Connect(&(Cascade_Inf->lower->sock),Cascade_Inf->lower->hostName,Cascade_Inf->lower->portNumber);
		
				if (status != STATUS_OK)
				{
					PGRset_cascade_server_status(Cascade_Inf->lower,DATA_ERR);
					Cascade_Inf->lower = NULL;
				}
			}
		}
		while (Cascade_Inf->lower == NULL)
		{
			Cascade_Inf->lower = PGRget_lower_cascade();	
			if (Cascade_Inf->lower == NULL)
			{
#ifdef PRINT_DEBUG
				show_debug("%s:there is no lower cascaded server",func);
#endif			
				return STATUS_ERROR;
			}
			status = PGR_Create_Socket_Connect(&(Cascade_Inf->lower->sock),Cascade_Inf->lower->hostName,Cascade_Inf->lower->portNumber);
	
			if (status != STATUS_OK)
			{
#ifdef PRINT_DEBUG
				show_debug("%s:PGRset_cascade_server_status call status ERROR",func);
#endif			
				PGRset_cascade_server_status(Cascade_Inf->lower,DATA_ERR);
				Cascade_Inf->lower = NULL;
			}
			else
			{
				break;
			}
		}
		if (PGRsend_cascade(Cascade_Inf->lower->sock,header,query) != STATUS_OK)
		{
			show_error("%s:PGRsend_cascade failed",func);
			if (Cascade_Inf->lower->sock > 0)
			{
				close(Cascade_Inf->lower->sock);
				Cascade_Inf->lower->sock = -1;
			}
			PGRset_cascade_server_status(Cascade_Inf->lower,DATA_ERR);
			Cascade_Inf->lower = NULL;
		}
		if (Cascade_Inf->lower != NULL)
		{
			break;
		}
	}
	return STATUS_OK;
}

int
PGRsend_upper_cascade(ReplicateHeader * header, char * query)
{
	char * func = "PGRsend_upper_cascade()";
	int status = STATUS_OK;
	int sock;

	for(;;)
	{
		if (Cascade_Inf->upper != NULL)
		{
			if (Cascade_Inf->upper->sock <= 0)
			{
				Cascade_Inf->upper = NULL;
			}
		}
		while (Cascade_Inf->upper == NULL)
		{
			Cascade_Inf->upper = PGRget_upper_cascade();	
			if (Cascade_Inf->upper == NULL)
			{
				show_debug("Cascade_Inf->upper is NULL");
				return STATUS_ERROR;
			}
#ifdef PRINT_DEBUG
			show_debug("%s:sock[%d] host[%s] port[%d]", func,sock,Cascade_Inf->upper->hostName,Cascade_Inf->upper->portNumber);
#endif			
			status = PGR_Create_Socket_Connect(&sock,Cascade_Inf->upper->hostName,Cascade_Inf->upper->portNumber);
	
			if (status != STATUS_OK)
			{
#ifdef PRINT_DEBUG
				show_debug("%s:PGR_Create_Socket_Connect failed",func);
#endif			
				PGRset_cascade_server_status(Cascade_Inf->upper,DATA_ERR);
				Cascade_Inf->upper = NULL;
			}
			else
			{
#ifdef PRINT_DEBUG
				show_debug("%s:PGR_Create_Socket_Connect OK[%d]",func,sock);
#endif			
				Cascade_Inf->upper->sock = sock;
				break;
			}
		}
		if (PGRsend_cascade(Cascade_Inf->upper->sock,header,query) != STATUS_OK)
		{
#ifdef PRINT_DEBUG
			show_debug("%s:PGRsend_cascade failed",func);
#endif			
			PGRset_cascade_server_status(Cascade_Inf->upper,DATA_ERR);
			Cascade_Inf->upper = NULL;
		}
		if (Cascade_Inf->upper != NULL)
		{
			break;
		}
	}
	return STATUS_OK;
}

ReplicateServerInfo *
PGRget_lower_cascade(void)
{
	char * func = "PGRget_lower_cascade()";
	ReplicateServerInfo * cascade = NULL;

	if ((Cascade_Tbl == NULL) || (Cascade_Inf == NULL))
	{
		show_error("%s:config data read error",func);
		return NULL;
	}

	/* count lower server */
	cascade = Cascade_Inf->myself;
	if (cascade == NULL)
	{
		show_error("%s:cascade data initialize error",func);
		return NULL;
	}
	if (cascade->useFlag != DATA_END)
	{
		cascade ++;
	}
	while (cascade->useFlag != DATA_END)
	{
#ifdef PRINT_DEBUG
		show_debug("%s:lower cascade search[%d]@[%s] use[%d]",
			func,
			cascade->portNumber,
			cascade->hostName,
			cascade->useFlag);
#endif			
		if (cascade->useFlag == DATA_USE)
		{
#ifdef PRINT_DEBUG
			show_debug("%s:find lower cascade",func);
#endif			
			return cascade;
		}
		cascade ++;
	}
	return NULL;
}

ReplicateServerInfo *
PGRget_upper_cascade(void)
{
	char * func = "PGRget_upper_cascade()";
	ReplicateServerInfo * cascade = NULL;

	if ((Cascade_Tbl == NULL) || (Cascade_Inf == NULL))
	{
		show_error("%s:config data read error",func);
		return NULL;
	}

	/* count lower server */
	cascade = Cascade_Inf->myself;
	if ((cascade == NULL) || (Cascade_Inf->top == cascade))
	{
		return NULL;
	}
	cascade --;
	while (cascade != NULL)
	{
		if (cascade->useFlag == DATA_USE)
		{
			return cascade;
		}
		if (Cascade_Inf->top == cascade)
		{
			break;
		}
		cascade --;
	}
	return NULL;
}

static void
write_cascade_status_file(ReplicateServerInfo * cascade)
{
	switch( cascade->useFlag)
	{
		case DATA_FREE:
			PGRwrite_log_file(Com_Info.Log_Info.StatusFp,"cascade(%s) port(%d) free",
					cascade->hostName,
					cascade->portNumber);
			break;
		case DATA_INIT:
			PGRwrite_log_file(Com_Info.Log_Info.StatusFp,"cascade(%s) port(%d) initialize",
					cascade->hostName,
					cascade->portNumber);
			break;
		case DATA_USE:
			PGRwrite_log_file(Com_Info.Log_Info.StatusFp,"cascade(%s) port(%d) start use",
					cascade->hostName,
					cascade->portNumber);
			break;
		case DATA_ERR:
			PGRwrite_log_file(Com_Info.Log_Info.StatusFp,"cascade(%s) port(%d) error",
					cascade->hostName,
					cascade->portNumber);
			break;
		case DATA_TOP:
			PGRwrite_log_file(Com_Info.Log_Info.StatusFp,"cascade(%s) port(%d) become top",
					cascade->hostName,
					cascade->portNumber);
			break;
	}
}

void
PGRset_cascade_server_status(ReplicateServerInfo * cascade, int status)
{
	if (cascade == NULL)
	{
		return;
	}
	if (cascade->useFlag != status)
	{
		cascade->useFlag = status;
		write_cascade_status_file(cascade);
	}
}

ReplicateServerInfo *
PGRrecv_cascade_answer(ReplicateServerInfo * cascade,ReplicateHeader * header)
{
	ReplicateServerInfo * answer = NULL;

	if ((cascade == NULL) || (header == NULL))
	{
		return NULL;
	}
	answer = (ReplicateServerInfo*)PGRread_packet(cascade->sock,header);
	return answer;
}

int
PGRsend_cascade(int sock , ReplicateHeader * header, char * query)
{
	char * func ="PGRsend_cascade()";
	int s;
	char * send_ptr;
	char * buf;
	int send_size = 0;
	int buf_size;
	int header_size;
	int rtn;
	fd_set      wmask;
	struct timeval timeout;
	int query_size = 0;

	/* check parameter */
	if ((header == NULL) || (sock <= 0))
	{
		show_error("%s:header or sock is null",func);
		return STATUS_ERROR;
	}

#ifdef PRINT_DEBUG
	show_debug("%s:PGRsend_cascade sock[%d]",func,sock);
#endif			
	query_size = ntohl(header->query_size);
	header_size = sizeof(ReplicateHeader);
	buf_size = header_size + query_size + 4;
	buf = malloc(buf_size);
	memset(buf,0,buf_size);
	buf_size -= 4;
	memcpy(buf,header,header_size);
	if (query_size > 0)
	{
		memcpy((char *)(buf+header_size),query,query_size+1);
	}
	send_ptr = buf;

	timeout.tv_sec = 10;
	timeout.tv_usec = 0;

	/*
	 * Wait for something to happen.
	 */
	FD_ZERO(&wmask);
	FD_SET(sock,&wmask);
	rtn = select(sock+1, (fd_set *)NULL, &wmask, (fd_set *)NULL, &timeout);
	if (rtn && FD_ISSET(sock, &wmask))
	{
		for (;;)
		{
			s = send(sock,send_ptr + send_size,buf_size - send_size ,0);
			if (s < 0){
				if (errno == EINTR)
				{
					continue;
				}
				show_error("%s:send failed",func);
				free(buf);
				return STATUS_ERROR;
			}
			if (s == 0)
			{
				free(buf);
				return STATUS_ERROR;
			}
			send_size += s;
			if (send_size == buf_size)
			{
#ifdef PRINT_DEBUG
				show_debug("%s:send[%s] size[%d]",func,query,send_size);
#endif			
				free(buf);
				return STATUS_OK;
			}
		}
	}
	return STATUS_OK;
}

int
PGRwait_answer_cascade(int  sock)
{
	ReplicateHeader header;
	char * answer = NULL;

	answer = PGRread_packet(sock,&header);
	if (answer != NULL)
	{
		free(answer);
		return STATUS_OK;
	}
	return STATUS_ERROR;
}

static int
notice_cascade_data(int sock)
{
	char * func = "notice_cascade_data";
	ReplicateServerInfo *cascade_data = NULL;
	ReplicateHeader header;
	int cnt = 0;
	int size = 0;

	if (sock <= 0)
	{
		return STATUS_ERROR;
	}

	cascade_data = get_cascade_data(&cnt, ALL_CASCADE );
	if (cnt <= 0)
	{
		show_error("%s:cascade data is wrong",func);
		return STATUS_ERROR;
	}
	size = sizeof (ReplicateServerInfo) * cnt ;

	memset(&header,0,sizeof(ReplicateHeader));
	header.cmdSys = CMD_SYS_CASCADE ;
	header.cmdSts = CMD_STS_TO_LOWER ;
	header.cmdType = CMD_TYPE_UPDATE_ALL;
	header.query_size = htonl(size);
	PGRsend_cascade(sock, &header, (char *)cascade_data );
	if (cascade_data != NULL)
	{
		free(cascade_data);
	}
	return STATUS_OK;
}

int
PGRcascade_main(int sock, ReplicateHeader * header, char * query)
{
	switch (header->cmdSts)
	{
		case CMD_STS_TO_UPPER:
			if (header->cmdType == CMD_TYPE_ADD)
			{
				/* add lower cascade data to myself */
				add_cascade_data(header,(ReplicateServerInfo*)query);
				/* send cascade data to upper */
				/* and receive new cascade data from upper */
				PGRstartup_cascade();
				/* return to lower with new cascade data */
				notice_cascade_data(sock);
				/* notifies a cascade server's information to Cluster DBs */
				notice_cascade_data_to_cluster_db();
			}
			break;
		case CMD_STS_TO_LOWER:
			/*
			 * use for cascading replication 
			 */
			break;
	}
	return STATUS_OK;
}

static int
notice_cascade_data_to_cluster_db(void)
{
	char userName[USERNAME_MAX_LENGTH];

	if (Cascade_Inf->lower == NULL)
	{
		Cascade_Inf->lower = PGRget_lower_cascade();	
	}
	if (Cascade_Inf->lower == NULL)
	{
		return STATUS_ERROR;
	}
	memset(userName,0,sizeof(userName));
	strncpy(userName ,getenv("LOGNAME"),sizeof(userName)-1);

	PGRnotice_replication_server(Cascade_Inf->lower->hostName, Cascade_Inf->lower->portNumber, Cascade_Inf->lower->recoveryPortNumber,Cascade_Inf->lower->lifecheckPortNumber,userName);

	return STATUS_OK;
}

int
PGRwait_notice_rlog_done(void)
{

	if (Cascade_Inf->lower != NULL)
	{
		if (Cascade_Inf->lower->sock > 0)
		{
			PGRreturn_result(Cascade_Inf->lower->sock,PGR_WAIT_ANSWER);
			return STATUS_OK;
		}
	}
	return STATUS_ERROR;
}

int
PGRsend_notice_rlog_done(int sock)
{
	ReplicateHeader header;
	int size = 0;

	if (sock <= 0)
	{
		return STATUS_ERROR;
	}

	size = strlen(PGR_QUERY_DONE_NOTICE_CMD);
	memset(&header,0,sizeof(ReplicateHeader));
	header.cmdSys = CMD_SYS_CASCADE ;
	header.cmdSts = CMD_STS_RESPONSE ;
	header.cmdType = 0;
	header.query_size = htonl(size);
	PGRsend_cascade(sock, &header, PGR_QUERY_DONE_NOTICE_CMD);
	return STATUS_OK;

}
#endif /* USE_REPLICATION */
