/*--------------------------------------------------------------------
 * FILE:
 *    conf.c
 *    Replication server for PostgreSQL
 *
 * NOTE:
 *    Read and set configuration data in this modul.
 *
 * Portions Copyright (c) 2003, Atsushi Mitani
 *--------------------------------------------------------------------
 */
#ifdef USE_REPLICATION

#include "postgres.h"

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <fcntl.h>
#include <ctype.h>
#include <sys/stat.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/sem.h>
#include <netdb.h>
#include <errno.h>
#include <sys/file.h>

#include "libpq-fe.h"
#include "replicate_com.h"
#include "pgreplicate.h"

int PGRset_Conf_Data(char * path);

/*--------------------------------------------------------------------
 * SYMBOL
 *    PGRset_Conf_Data()
 * NOTES
 *    Initialize mamory and tables
 * ARGS
 *    char * path: path of the setup file (I)
 * RETURN
 *    OK: STATUS_OK
 *    NG: STATUS_ERROR
 *--------------------------------------------------------------------
 */
int
PGRset_Conf_Data(char * path)
{
	char * func = "init_server_tbl()";
	HostTbl host_tbl[MAX_DB_SERVER];
	ConfDataType * conf = NULL;
	int cnt = 0;
	int lb_cnt = 0;
	int cascade_cnt = 0;
	int rec_no = 0;
	int lb_rec_no = 0;
	int cascade_rec_no = -1;
	int i = 0;
	int size = 0;
	char fname[256];
	union semun sem_arg;

	/*
	 * open log file
	 */
	if (path == NULL)
	{
		path = ".";
	}

	if (Com_Info.Log_Info.Log_Print)
	{
		snprintf(fname,sizeof(fname),"%s/%s",path,PGREPLICATE_LOG_FILE);
		Com_Info.Log_Info.LogFp = fopen(fname,"a");
		if (Com_Info.Log_Info.LogFp == NULL)
		{
			show_error("%s:fopen failed: (%s)",func,strerror(errno));
			return STATUS_ERROR;
		}
	}

	snprintf(fname,sizeof(fname),"%s/%s",path,PGREPLICATE_STATUS_FILE);
	Com_Info.Log_Info.StatusFp = fopen(fname,"a");
	if (Com_Info.Log_Info.StatusFp == NULL)
	{
		show_error("%s:fopen failed: (%s)",func,strerror(errno));
		return STATUS_ERROR;
	}

	/*
	 * Recovery Queue structure initialize
	 */
	RecoveryQueue.queue_fp = (FILE *)NULL;
	RecoveryQueue.current_queue_no = 0;

	/*
	 * read configuration file
	 */
	if (PGR_Get_Conf_Data(path,PGREPLICATE_CONF_FILE) != STATUS_OK)
	{
		show_error("%s:PGR_Get_Conf_Data failed",func);
		return STATUS_ERROR;
	}

	/* allocate response information table */
	PGR_Response_Inf = (ResponseInf *)malloc(sizeof(ResponseInf));
	if (PGR_Response_Inf == NULL)
	{
		show_error("%s:malloc() failed. reason: %s", func,strerror(errno));
		return STATUS_ERROR;
	}
	PGR_Response_Inf->response_mode = PGR_NORMAL_MODE;
	PGR_Response_Inf->current_cluster = 0;

	/*
	 * memory allocate load balance table buffer
	 */
	size = sizeof(RecoveryTbl) * MAX_DB_SERVER;
	LoadBalanceTblShmid = shmget(IPC_PRIVATE,size,IPC_CREAT | IPC_EXCL | 0600);
	if (LoadBalanceTblShmid < 0)
	{
		show_error("shmget() failed. reason: %s", strerror(errno));
		return STATUS_ERROR;
	}
	LoadBalanceTbl = (RecoveryTbl *)shmat(LoadBalanceTblShmid,0,0);
	if (LoadBalanceTbl == (RecoveryTbl *)-1)
	{
		show_error("%s:shmat() failed. reason: %s", func,strerror(errno));
		return STATUS_ERROR;
	}
	if (LoadBalanceTbl == (RecoveryTbl *)NULL)
	{
		show_error("%s:malloc failed: (%s)",func,strerror(errno));
		return STATUS_ERROR;
	}
	memset(LoadBalanceTbl , 0 , size );

	/*
	 * memory allocate cascade server table buffer
	 */
	size = sizeof(ReplicateServerInfo) * MAX_DB_SERVER;
	CascadeTblShmid = shmget(IPC_PRIVATE,size,IPC_CREAT | IPC_EXCL | 0600);
	if (CascadeTblShmid < 0)
	{
		show_error("%s:shmget() failed. reason: %s", func,strerror(errno));
		return STATUS_ERROR;
	}
	Cascade_Tbl = (ReplicateServerInfo *)shmat(CascadeTblShmid,0,0);
	if (Cascade_Tbl == (ReplicateServerInfo *)-1)
	{
		show_error("%s:shmat() failed. reason: %s",func, strerror(errno));
		return STATUS_ERROR;
	}
	memset(Cascade_Tbl , 0 , size );

	/*
	 * memory allocate cascade index 
	 */
	size = sizeof(CascadeInf);
	CascadeInfShmid = shmget(IPC_PRIVATE,size,IPC_CREAT | IPC_EXCL | 0600);
	if (CascadeInfShmid < 0)
	{
		show_error("%s:shmget() failed. reason: %s", func,strerror(errno));
		return STATUS_ERROR;
	}
	Cascade_Inf = (CascadeInf *)shmat(CascadeInfShmid,0,0);
	if (Cascade_Inf == (CascadeInf *)-1)
	{
		show_error("%s:shmat() failed. reason: %s", func,strerror(errno));
		return STATUS_ERROR;
	}
	memset(Cascade_Inf , 0 , size );

	/*
	 * memory allocate replication commit log buffer
	 */
	size = sizeof(CommitLogInf) * MAX_DB_SERVER * MAX_CONNECTIONS;
	CommitLogShmid = shmget(IPC_PRIVATE,size,IPC_CREAT | IPC_EXCL | 0600);
	if (CommitLogShmid < 0)
	{
		show_error("%s:shmget() failed. reason: %s", func,strerror(errno));
		return STATUS_ERROR;
	}
	Commit_Log_Tbl = (CommitLogInf *)shmat(CommitLogShmid,0,0);
	if (Commit_Log_Tbl == (CommitLogInf *)-1)
	{
		show_error("%s:shmat() failed. reason: %s",func, strerror(errno));
		return STATUS_ERROR;
	}
	memset(Commit_Log_Tbl , 0 , size );
	(Commit_Log_Tbl + (MAX_DB_SERVER * MAX_CONNECTIONS) -1)->inf.useFlag = DATA_END;

	/* create semapho */
	if ((SemID = semget(IPC_PRIVATE,4,IPC_CREAT | IPC_EXCL | 0600)) < 0)
	{
		show_error("%s:semget() failed. (%s)",func,strerror(errno));
		return STATUS_ERROR;
	}
	for ( i = 0 ; i < 4 ; i ++)
	{
		semctl(SemID, i, GETVAL, sem_arg);
		sem_arg.val = 1;
		semctl(SemID, i, SETVAL, sem_arg);
	}

	size = sizeof(ReplicationLogInf);
	Replicateion_Log = malloc(size);
	if (Replicateion_Log == NULL)
	{
		show_error("%s:malloc failed: (%s)",func,strerror(errno));
		return STATUS_ERROR;
	}
	memset(Replicateion_Log , 0 , size );
	Replicateion_Log->RLog_Sock_Path = NULL;

	/*
	 * set each datas into the tables
	 */
	conf = Com_Info.ConfData_Info.ConfData_Top;
	while (conf != (ConfDataType *)NULL) 
	{
		/* get cluster db data */
		if (!strcmp(conf->table,CLUSTER_SERVER_TAG))
		{
			rec_no = conf->rec_no;
			if (cnt < rec_no)
			{
				cnt = rec_no;
				if (cnt >= MAX_DB_SERVER)
				{
					continue;
				}
			}
			if (!strcmp(conf->key,HOST_NAME_TAG))
			{
				strncpy(host_tbl[rec_no].hostName,conf->value,sizeof(host_tbl[rec_no].hostName));
				conf = (ConfDataType*)conf->next;
				continue;
			}
			if (!strcmp(conf->key,PORT_TAG))
			{
				host_tbl[rec_no].port = atoi(conf->value);
				conf = (ConfDataType*)conf->next;
				continue;
			}
			if (!strcmp(conf->key,RECOVERY_PORT_TAG))
			{
				host_tbl[rec_no].recoveryPort = atoi(conf->value);
				conf = (ConfDataType*)conf->next;
				continue;
			}
			if (!STRCMP(conf->key,LIFECHECK_PORT_TAG))
			{
				if (atoi(conf->value) > 0)
				{
					host_tbl[rec_no].lifecheckPort = atoi(conf->value);
				}
				else
				{
					host_tbl[rec_no].lifecheckPort = DEFAULT_CLUSTER_LIFECHECK_PORT;
				}
				conf = (ConfDataType*)conf->next;
				continue;
			}
		}
		/* get cascade server data */
		else if (!strcmp(conf->table, REPLICATION_SERVER_INFO_TAG))
		{
			cascade_rec_no = conf->rec_no ;
			if (cascade_cnt < cascade_rec_no)
			{
				cascade_cnt = cascade_rec_no;
				if (cascade_cnt >= MAX_DB_SERVER)
				{
					continue;
				}
			}
			if (!strcmp(conf->key,HOST_NAME_TAG))
			{
				strncpy((Cascade_Tbl+cascade_rec_no)->hostName,conf->value,sizeof(Cascade_Tbl->hostName));
				conf = (ConfDataType*)conf->next;
				continue;
			}
			if (!strcmp(conf->key,PORT_TAG))
			{
				if (atoi(conf->value) > 0)
				{
					(Cascade_Tbl+cascade_rec_no)->portNumber = atoi(conf->value);
				}
				else
				{
					(Cascade_Tbl+cascade_rec_no)->portNumber = DEFAULT_PGRP_PORT;
				}
				conf = (ConfDataType*)conf->next;
				PGRset_cascade_server_status(Cascade_Tbl+cascade_rec_no,DATA_USE);
				if (cascade_rec_no == 0)
				{
					Cascade_Inf->top = Cascade_Tbl;
				}
				continue;
			}
			if (!strcmp(conf->key,RECOVERY_PORT_TAG))
			{
				if (atoi(conf->value) > 0)
				{
					(Cascade_Tbl+cascade_rec_no)->recoveryPortNumber = atoi(conf->value);
				}
				else
				{
					(Cascade_Tbl+cascade_rec_no)->recoveryPortNumber = DEFAULT_PGRP_RECOVERY_PORT;
				}
				conf = (ConfDataType*)conf->next;
				continue;
			}
			if (!STRCMP(conf->key,LIFECHECK_PORT_TAG))
			{
				if (atoi(conf->value) > 0)
				{
					(Cascade_Tbl+cascade_rec_no)->lifecheckPortNumber = atoi(conf->value);
				}
				else
				{
					(Cascade_Tbl+cascade_rec_no)->lifecheckPortNumber = DEFAULT_PGRP_LIFECHECK_PORT;
				}
				conf = (ConfDataType*)conf->next;
				continue;
			}
		}
		/* get loadbalancer table data */
		else if (!strcmp(conf->table,LOAD_BALANCE_SERVER_TAG))
		{
			lb_rec_no = conf->rec_no;
			if (lb_cnt < lb_rec_no)
			{
				lb_cnt = lb_rec_no;
				if (lb_cnt >= MAX_DB_SERVER)
				{
					continue;
				}
			}
			if (!strcmp(conf->key,HOST_NAME_TAG))
			{
				strncpy((LoadBalanceTbl + lb_rec_no)->hostName, conf->value,sizeof(LoadBalanceTbl->hostName));
				conf = (ConfDataType*)conf->next;
				continue;
			}
			if (!strcmp(conf->key,RECOVERY_PORT_TAG))
			{
				(LoadBalanceTbl + lb_rec_no)->useFlag = DATA_USE; 
				(LoadBalanceTbl + lb_rec_no+1)->useFlag = DATA_END; 
				if (atoi(conf->value) > 0)
				{
					(LoadBalanceTbl + lb_rec_no)->recoveryPort = atoi(conf->value);
				}
				else
				{
					(LoadBalanceTbl + lb_rec_no)->recoveryPort = DEFAULT_PGLB_RECOVERY_PORT;
				}
				conf = (ConfDataType*)conf->next;
				continue;
			}
			if (!STRCMP(conf->key,LIFECHECK_PORT_TAG))
			{
				if (atoi(conf->value) > 0)
				{
					(LoadBalanceTbl + lb_rec_no)->lifecheckPort = atoi(conf->value);
				}
				else
				{
					(LoadBalanceTbl + lb_rec_no)->lifecheckPort = DEFAULT_PGLB_LIFECHECK_PORT;
				}
				conf = (ConfDataType*)conf->next;
				continue;
			}
		}
		else
		{
			if (!strcmp(conf->key,REPLICATE_PORT_TAG))
			{
				if (atoi(conf->value) > 0)
				{
					Port_Number = atoi(conf->value);
				}
				else
				{
					Port_Number =DEFAULT_PGRP_PORT;
				}
				conf = (ConfDataType*)conf->next;
				continue;
			}
			if (!strcmp(conf->key,RECOVERY_PORT_TAG))
			{
				if (atoi(conf->value) > 0)
				{
					Recovery_Port_Number = atoi(conf->value);
				}
				else
				{
					Recovery_Port_Number =DEFAULT_PGRP_RECOVERY_PORT;
				}
				conf = (ConfDataType*)conf->next;
				continue;
			}
			if (!STRCMP(conf->key,LIFECHECK_PORT_TAG))
			{
				if (atoi(conf->value) > 0)
				{
					LifeCheck_Port_Number = atoi(conf->value);
				}
				else
				{
					LifeCheck_Port_Number = DEFAULT_PGRP_LIFECHECK_PORT;
				}
				conf = (ConfDataType*)conf->next;
				continue;
			}
			if (!STRCMP(conf->key,RLOG_PORT_TAG))
			{
				if (atoi(conf->value) > 0)
				{
					Replicateion_Log->RLog_Port_Number = atoi(conf->value);
				}
				else
				{
					Replicateion_Log->RLog_Port_Number = DEFAULT_PGRP_RLOG_PORT;
				}
				conf = (ConfDataType*)conf->next;
				continue;
			}
			if (!STRCMP(conf->key,STATUS_LOG_FILE_TAG))
			{
				Com_Info.Log_Info.PGRStatusFileName = strdup(conf->value);
				conf = (ConfDataType*)conf->next;
				continue;
			}
			if (!STRCMP(conf->key,ERROR_LOG_FILE_TAG))
			{
				Com_Info.Log_Info.PGRLogFileName = strdup(conf->value);
				conf = (ConfDataType*)conf->next;
				continue;
			}
			/* get response mode */
			if (!strcmp(conf->key,RESPONSE_MODE_TAG))
			{
				if (!strcmp(conf->value,RESPONSE_MODE_RELIABLE))
				{
					PGR_Response_Inf->response_mode = PGR_RELIABLE_MODE;
				}
				else if (!strcmp(conf->value,RESPONSE_MODE_FAST))
				{
					PGR_Response_Inf->response_mode = PGR_FAST_MODE;
				}
				else
				{
					PGR_Response_Inf->response_mode = PGR_NORMAL_MODE;
				}
				conf = (ConfDataType*)conf->next;
				continue;
			}
			/* get replication log use or not */
			if (!strcmp(conf->key,USE_REPLICATION_LOG_TAG))
			{
				if (!strcmp(conf->value,"yes"))
				{
					PGR_Use_Replication_Log = true;
				}
				conf = (ConfDataType*)conf->next;
				continue;
			}
			/* get reserved connection number */
			if (!strcmp(conf->key,RESERVED_CONNECTIONS_TAG))
			{
				if (atoi(conf->value) > 1)
				{
					Reserved_Connections = atoi(conf->value);
				}
				else
				{
					Reserved_Connections = 1;
				}
				conf = (ConfDataType*)conf->next;
				continue;
			}
			conf = (ConfDataType*)conf->next;
		}
		conf = (ConfDataType*)conf->next;
	}

	/*
	 * open log file
	 */
	if (Com_Info.Log_Info.Log_Print)
	{
		if (Com_Info.Log_Info.PGRLogFileName == NULL)
		{
			snprintf(fname,sizeof(fname),"%s/%s",path,PGREPLICATE_LOG_FILE);
		}
		else
		{
			strncpy(fname,Com_Info.Log_Info.PGRLogFileName,sizeof(fname));
		}
		Com_Info.Log_Info.LogFp = fopen(fname,"a");
		if (Com_Info.Log_Info.LogFp == NULL)
		{
			show_error("%s:fopen failed: (%s)",func,strerror(errno));
			return STATUS_ERROR;
		}
	}

	if (Com_Info.Log_Info.PGRStatusFileName == NULL)
	{
		snprintf(fname,sizeof(fname),"%s/%s",path,PGREPLICATE_STATUS_FILE);
	}
	else
	{
		strncpy(fname,Com_Info.Log_Info.PGRStatusFileName,sizeof(fname));
	}
	Com_Info.Log_Info.StatusFp = fopen(fname,"a");
	if (Com_Info.Log_Info.StatusFp == NULL)
	{
		show_error("%s:fopen failed: (%s)",func,strerror(errno));
		return STATUS_ERROR;
	}

	/* create cluster db server table */
	Host_Tbl_Begin = (HostTbl *)NULL;

	size = sizeof(HostTbl) * MAX_DB_SERVER;
	HostTblShmid = shmget(IPC_PRIVATE,size,IPC_CREAT | IPC_EXCL | 0600);
	if (HostTblShmid < 0)
	{
		show_error("%s:shmget() failed. reason: %s",func, strerror(errno));
		return STATUS_ERROR;
	}
	Host_Tbl_Begin = (HostTbl *)shmat(HostTblShmid,0,0);
	if (Host_Tbl_Begin == (HostTbl *)-1)
	{
		show_error("%s:shmat() failed. reason: %s",func, strerror(errno));
		return STATUS_ERROR;
	}
	memset(Host_Tbl_Begin , 0 , size );
	Host_Tbl_Begin -> useFlag = DATA_END;

	for ( i = 0 ; i <= cnt ; i ++)
	{
		PGRadd_HostTbl(&host_tbl[i],DATA_USE);
	}
	/* set load balance table */
	for ( i = 0 ; i <= lb_cnt ; i ++)
	{
		(LoadBalanceTbl + i)->port = -1;
		(LoadBalanceTbl + i)->sock = -1;
	}
	memset((LoadBalanceTbl + i),0,sizeof(RecoveryTbl));
	PGR_Free_Conf_Data();

	/* allocate result buffer of query */
	PGR_Result = malloc(PGR_MESSAGE_BUFSIZE);
	if (PGR_Result == NULL)
	{
		show_error("%s:malloc() failed. reason: %s",func,strerror(errno));
		return STATUS_ERROR;
	}
	memset(PGR_Result,0,PGR_MESSAGE_BUFSIZE);

	/* allocate log_data */
	PGR_Log_Header = malloc(sizeof(ReplicateHeader));
	if (PGR_Log_Header == NULL)
	{
		show_error("%s:malloc() failed. reason: %s",func, strerror(errno));
		return STATUS_ERROR;
	}
	memset(PGR_Log_Header,0,sizeof(ReplicateHeader));

	/* allocate send query id */
	size = sizeof(unsigned int) * (MAX_DB_SERVER +1);
	PGR_Send_Query_ID = malloc (sizeof(unsigned int) * MAX_DB_SERVER);
	if (PGR_Send_Query_ID == NULL)
	{
		show_error("%s:malloc() failed. reason: %s",func, strerror(errno));
		return STATUS_ERROR;
	}
	memset(PGR_Send_Query_ID, 0, size);

	/* create lock wait table */
	size = sizeof(LockWaitInf);
	LockWaitTblShmid = shmget(IPC_PRIVATE,size,IPC_CREAT | IPC_EXCL | 0600);
	if (LockWaitTblShmid < 0)
	{
		show_error("%s:shmget() failed. reason: %s", func,strerror(errno));
		return STATUS_ERROR;
	}
	Lock_Wait_Tbl = (LockWaitInf *)shmat(LockWaitTblShmid,0,0);
	if (Lock_Wait_Tbl == (LockWaitInf *)-1)
	{
		show_error("%s:shmat() failed. reason: %s",func, strerror(errno));
		return STATUS_ERROR;
	}
	memset(Lock_Wait_Tbl , 0 , size );
	
	for ( i = 0 ; i < MAX_DB_SERVER ; i ++)
	{
		StartReplication[i] = true;
	}

	/* set self data into cascade table */

	cascade_rec_no ++;
	gethostname((Cascade_Tbl+cascade_rec_no)->hostName,sizeof(Cascade_Tbl->hostName));
	(Cascade_Tbl+cascade_rec_no)->portNumber = Port_Number;
	(Cascade_Tbl+cascade_rec_no)->recoveryPortNumber = Com_Info.ConfData_Info.Recovery_Port_Number;

	PGRset_cascade_server_status(Cascade_Tbl+cascade_rec_no,DATA_USE);
	/* terminate */
	(Cascade_Tbl+(cascade_rec_no+1))->useFlag = DATA_END;

	Cascade_Inf->top = Cascade_Tbl;
	Cascade_Inf->end = Cascade_Tbl+cascade_rec_no;
	Cascade_Inf->lower = NULL;
	if (cascade_rec_no >= 1)
	{
		Cascade_Inf->upper = (Cascade_Tbl+cascade_rec_no - 1);
	}

	Cascade_Inf->myself = (Cascade_Tbl+cascade_rec_no);
	Cascade_Inf->useFlag = DATA_USE;

	return STATUS_OK;
}

#endif /* USE_REPLICATION */
