/*
 *	controldata.c
 *
 *	controldata functions
 */

#include "pg_migrator.h"

#include <ctype.h>
#include <stdlib.h>

#ifdef EDB_NATIVE_LANG
#include "access/tuptoaster.h"
#endif

/*
 * get_control_data()
 *
 * gets pg_control information in "ctrl". Assumes that bindir and
 * datadir are valid absolute paths to postgresql bin and pgdata
 * directories respectively *and* pg_resetxlog is version compatible
 * with datadir. The main purpose of this function is to get pg_control
 * data in a version independent manner.
 *
 * The approach taken here is to invoke pg_resetxlog with -n option
 * and then pipe its output. With little string parsing we get the
 * pg_control data.
 */
void
get_control_data(migratorContext *ctx, ClusterInfo *cluster)
{
	char		cmd[MAXPGPATH];
	char		bufin[MAX_STRING];
	FILE	   *output;
	char	   *p;
	bool		got_xid = false;
	bool		got_oid = false;
	bool		got_log_id = false;
	bool		got_log_seg = false;
	bool		got_tli = false;
	bool		got_align = false;
	bool		got_blocksz = false;
	bool		got_largesz = false;
	bool		got_walsz = false;
	bool		got_walseg = false;
	bool		got_ident = false;
	bool		got_index = false;
	bool		got_toast = false;
	bool		got_date_is_int = false;
	bool		got_float8_pass_by_value = false;
	char	   *lang = NULL;

	/* Because we test the pg_resetxlog output strings, it has to be in English. */
	if (getenv("LANG"))
		lang = pg_strdup(ctx, getenv("LANG"));
#ifndef WIN32
	putenv(pg_strdup(ctx, "LANG=C"));
#else
	SetEnvironmentVariableA("LANG", "C");
#endif
	sprintf(cmd, SYSTEMQUOTE "\"%s/pg_resetxlog\" -n \"%s\"" SYSTEMQUOTE,
			cluster->bindir, cluster->pgdata);
	fflush(stdout);
	fflush(stderr);

	if ((output = popen(cmd, "r")) == NULL)
		pg_log(ctx, PG_FATAL, "Could not get control data: %s\n",
			   strerror(errno));

	/* Only pre-8.4 has these so if they are not set below we will check later */
	cluster->controldata.lc_collate = NULL;
	cluster->controldata.lc_ctype = NULL;

	/* Only in <= 8.3 */
	if (GET_MAJOR_VERSION(cluster->pg_version) <= 803)
	{
		cluster->controldata.float8_pass_by_value = false;
		got_float8_pass_by_value = true;
	}

#ifdef EDB_NATIVE_LANG
	/* EDB AS 8.3 is an 8.2 code base */
	if (cluster->is_edb_as && GET_MAJOR_VERSION(cluster->pg_version) <= 803)
	{
		cluster->controldata.toast = TOAST_MAX_CHUNK_SIZE;
		got_toast = true;
	}
#endif

	/* we have the result of cmd in "output". so parse it line by line now */
	while (fgets(bufin, sizeof(bufin), output))
	{
		if (ctx->debug)
			fprintf(ctx->debug_fd, bufin);

#ifdef WIN32
		/*
		 *	Due to an installer bug, LANG=C doesn't work for PG 8.3.3,
		 *	but does work 8.2.6 and 8.3.7, so check for non-ASCII output
		 *	and suggest a minor upgrade.
		 */
		if (GET_MAJOR_VERSION(pg_version) == 803)
		{
			for (p = bufin; *p; p++)
				if (!isascii(*p))
					pg_log(ctx, PG_FATAL,
						"The 8.3 cluster's pg_controldata is incapable of outputting ASCII, even\n"
						"with LANG=C.  You must upgrade this cluster to a newer version of Postgres\n"
						"8.3 to fix this bug.  Postgres 8.3.7 and later are known to work properly.\n");
		}
#endif

		if ((p = strstr(bufin, "pg_control version number:")) != NULL)
		{
			p = strchr(p, ':');

			if (p == NULL || strlen(p) <= 1)
				pg_log(ctx, PG_FATAL, "%d: pg_resetxlog  problem\n", __LINE__);

			p++;				/* removing ':' char */
			cluster->controldata.ctrl_ver = (uint32) atol(p);
		}
		else if ((p = strstr(bufin, "Catalog version number:")) != NULL)
		{
			p = strchr(p, ':');

			if (p == NULL || strlen(p) <= 1)
				pg_log(ctx, PG_FATAL, "%d: pg_resetxlog  problem\n", __LINE__);

			p++;				/* removing ':' char */
			cluster->controldata.cat_ver = (uint32) atol(p);
		}
		else if ((p = strstr(bufin, "First log file ID after reset:")) != NULL ||
				 (cluster->is_edb_as && GET_MAJOR_VERSION(cluster->pg_version) <= 803 &&
				  (p = strstr(bufin, "Current log file ID:")) != NULL))
		{
			p = strchr(p, ':');

			if (p == NULL || strlen(p) <= 1)
				pg_log(ctx, PG_FATAL, "%d: pg_resetxlog  problem\n", __LINE__);

			p++;				/* removing ':' char */
			cluster->controldata.logid = (uint32) atol(p);
			got_log_id = true;
		}
		else if ((p = strstr(bufin, "First log file segment after reset:")) != NULL ||
				 (cluster->is_edb_as && GET_MAJOR_VERSION(cluster->pg_version) <= 803 &&
				  (p = strstr(bufin, "Next log file segment:")) != NULL))
		{
			p = strchr(p, ':');

			if (p == NULL || strlen(p) <= 1)
				pg_log(ctx, PG_FATAL, "%d: pg_resetxlog  problem\n", __LINE__);

			p++;				/* removing ':' char */
			cluster->controldata.nxtlogseg = (uint32) atol(p);
			got_log_seg = true;
		}
		else if ((p = strstr(bufin, "Latest checkpoint's TimeLineID:")) != NULL)
		{
			p = strchr(p, ':');

			if (p == NULL || strlen(p) <= 1)
				pg_log(ctx, PG_FATAL, "%d: pg_resetxlog  problem\n", __LINE__);

			p++;				/* removing ':' char */
			cluster->controldata.chkpnt_tli = (uint32) atol(p);
			got_tli = true;
		}
		else if ((p = strstr(bufin, "Latest checkpoint's NextXID:")) != NULL)
		{
			char	   *op = strchr(p, '/');

			if (op == NULL)
				op = strchr(p, ':');

			if (op == NULL || strlen(op) <= 1)
				pg_log(ctx, PG_FATAL, "%d: pg_resetxlog  problem\n", __LINE__);

			op++;				/* removing ':' char */
			cluster->controldata.chkpnt_nxtxid = (uint32) atol(op);
			got_xid = true;
		}
		else if ((p = strstr(bufin, "Latest checkpoint's NextOID:")) != NULL)
		{
			p = strchr(p, ':');

			if (p == NULL || strlen(p) <= 1)
				pg_log(ctx, PG_FATAL, "%d: pg_resetxlog  problem\n", __LINE__);

			p++;				/* removing ':' char */
			cluster->controldata.chkpnt_nxtoid = (uint32) atol(p);
			got_oid = true;
		}
		else if ((p = strstr(bufin, "Maximum data alignment:")) != NULL)
		{
			p = strchr(p, ':');

			if (p == NULL || strlen(p) <= 1)
				pg_log(ctx, PG_FATAL, "%d: pg_resetxlog  problem\n", __LINE__);

			p++;				/* removing ':' char */
			cluster->controldata.align = (uint32) atol(p);
			got_align = true;
		}
		else if ((p = strstr(bufin, "Database block size:")) != NULL)
		{
			p = strchr(p, ':');

			if (p == NULL || strlen(p) <= 1)
				pg_log(ctx, PG_FATAL, "%d: pg_resetxlog  problem\n", __LINE__);

			p++;				/* removing ':' char */
			cluster->controldata.blocksz = (uint32) atol(p);
			got_blocksz = true;
		}
		else if ((p = strstr(bufin, "Blocks per segment of large relation:")) != NULL)
		{
			p = strchr(p, ':');

			if (p == NULL || strlen(p) <= 1)
				pg_log(ctx, PG_FATAL, "%d: pg_resetxlog  problem\n", __LINE__);

			p++;				/* removing ':' char */
			cluster->controldata.largesz = (uint32) atol(p);
			got_largesz = true;
		}
		else if ((p = strstr(bufin, "WAL block size:")) != NULL)
		{
			p = strchr(p, ':');

			if (p == NULL || strlen(p) <= 1)
				pg_log(ctx, PG_FATAL, "%d: pg_resetxlog  problem\n", __LINE__);

			p++;				/* removing ':' char */
			cluster->controldata.walsz = (uint32) atol(p);
			got_walsz = true;
		}
		else if ((p = strstr(bufin, "Bytes per WAL segment:")) != NULL)
		{
			p = strchr(p, ':');

			if (p == NULL || strlen(p) <= 1)
				pg_log(ctx, PG_FATAL, "%d: pg_resetxlog  problem\n", __LINE__);

			p++;				/* removing ':' char */
			cluster->controldata.walseg = (uint32) atol(p);
			got_walseg = true;
		}
		else if ((p = strstr(bufin, "Maximum length of identifiers:")) != NULL)
		{
			p = strchr(p, ':');

			if (p == NULL || strlen(p) <= 1)
				pg_log(ctx, PG_FATAL, "%d: pg_resetxlog  problem\n", __LINE__);

			p++;				/* removing ':' char */
			cluster->controldata.ident = (uint32) atol(p);
			got_ident = true;
		}
		else if ((p = strstr(bufin, "Maximum columns in an index:")) != NULL)
		{
			p = strchr(p, ':');

			if (p == NULL || strlen(p) <= 1)
				pg_log(ctx, PG_FATAL, "%d: pg_resetxlog  problem\n", __LINE__);

			p++;				/* removing ':' char */
			cluster->controldata.index = (uint32) atol(p);
			got_index = true;
		}
		else if ((p = strstr(bufin, "Maximum size of a TOAST chunk:")) != NULL)
		{
			p = strchr(p, ':');

			if (p == NULL || strlen(p) <= 1)
				pg_log(ctx, PG_FATAL, "%d: pg_resetxlog  problem\n", __LINE__);

			p++;				/* removing ':' char */
			cluster->controldata.toast = (uint32) atol(p);
			got_toast = true;
		}
		else if ((p = strstr(bufin, "Date/time type storage:")) != NULL)
		{
			p = strchr(p, ':');

			if (p == NULL || strlen(p) <= 1)
				pg_log(ctx, PG_FATAL, "%d: pg_resetxlog  problem\n", __LINE__);

			p++;				/* removing ':' char */
			cluster->controldata.date_is_int = strstr(p, "64-bit integers") != NULL;
			got_date_is_int = true;
		}
		else if ((p = strstr(bufin, "Float8 argument passing:")) != NULL)
		{
			p = strchr(p, ':');

			if (p == NULL || strlen(p) <= 1)
				pg_log(ctx, PG_FATAL, "%d: pg_resetxlog  problem\n", __LINE__);

			p++;				/* removing ':' char */
			/* used later for /contrib check */
			cluster->controldata.float8_pass_by_value = strstr(p, "by value") != NULL;
			got_float8_pass_by_value = true;
		}
		/* In pre-8.4 only */
		else if ((p = strstr(bufin, "LC_COLLATE:")) != NULL)
		{
			p = strchr(p, ':');

			if (p == NULL || strlen(p) <= 1)
				pg_log(ctx, PG_FATAL, "%d: pg_resetxlog  problem\n", __LINE__);

			p++;				/* removing ':' char */
			/* skip leading spaces and remove trailing newline */
			p += strspn(p, " ");
			if (strlen(p) > 0 && *(p + strlen(p) - 1) == '\n')
				*(p + strlen(p) - 1) = '\0';
			cluster->controldata.lc_collate = pg_strdup(ctx, p);
		}
		/* In pre-8.4 only */
		else if ((p = strstr(bufin, "LC_CTYPE:")) != NULL)
		{
			p = strchr(p, ':');

			if (p == NULL || strlen(p) <= 1)
				pg_log(ctx, PG_FATAL, "%d: pg_resetxlog  problem\n", __LINE__);

			p++;				/* removing ':' char */
			/* skip leading spaces and remove trailing newline */
			p += strspn(p, " ");
			if (strlen(p) > 0 && *(p + strlen(p) - 1) == '\n')
				*(p + strlen(p) - 1) = '\0';
			cluster->controldata.lc_ctype = pg_strdup(ctx, p);
		}
	}

	if (output)
		pclose(output);

	/* restore LANG */
	if (lang)
	{
#ifndef WIN32
		char   *envstr = (char *) pg_malloc(ctx, strlen(lang) + 6);

		sprintf(envstr, "LANG=%s", lang);
		putenv(envstr);
#else
		SetEnvironmentVariableA("LANG", lang);
#endif
		pg_free(lang);
	}
	else
	{
#ifndef WIN32
		unsetenv("LANG");
#else
		SetEnvironmentVariableA("LANG", "");
#endif
	}

	/* verify that we got all the mandatory pg_control data */
	if (!got_xid || !got_oid || !got_log_id || !got_log_seg || !got_tli ||
		!got_align || !got_blocksz || !got_largesz || !got_walsz ||
		!got_walseg || !got_ident || !got_index || !got_toast ||
		!got_date_is_int || !got_float8_pass_by_value)
	{
		pg_log(ctx, PG_REPORT,
			"Some required control information is missing;  cannot find:\n");

		if (!got_xid)
			pg_log(ctx, PG_REPORT, "  checkpoint next XID\n");

		if (!got_oid)
			pg_log(ctx, PG_REPORT, "  latest checkpoint next OID\n");

		if (!got_log_id)
			pg_log(ctx, PG_REPORT, "  first log file ID after reset\n");

		if (!got_log_seg)
			pg_log(ctx, PG_REPORT, "  first log file segment after reset\n");

		if (!got_tli)
			pg_log(ctx, PG_REPORT, "  latest checkpoint timeline ID\n");

		if (!got_align)
			pg_log(ctx, PG_REPORT, "  maximum alignment\n");

		if (!got_blocksz)
			pg_log(ctx, PG_REPORT, "  block size\n");

		if (!got_largesz)
			pg_log(ctx, PG_REPORT, "  large relation segment size\n");

		if (!got_walsz)
			pg_log(ctx, PG_REPORT, "  WAL block size\n");

		if (!got_walseg)
			pg_log(ctx, PG_REPORT, "  WAL segment size\n");

		if (!got_ident)
			pg_log(ctx, PG_REPORT, "  maximum identifier length\n");

		if (!got_index)
			pg_log(ctx, PG_REPORT, "  maximum number of indexed columns\n");

		if (!got_toast)
			pg_log(ctx, PG_REPORT, "  maximum TOAST chunk size\n");

		if (!got_date_is_int)
			pg_log(ctx, PG_REPORT, "  dates/times are integers?\n");

		/* value added in Postgres 8.4 */
		if (!got_float8_pass_by_value)
			pg_log(ctx, PG_REPORT, "  float8 argument passing method\n");

		pg_log(ctx, PG_FATAL,
			   "Unable to continue without required control information, terminating\n");
	}
}


/*
 * check_control_data()
 *
 * check to make sure the control data settings are compatible
 */
void
check_control_data(migratorContext *ctx, ControlData *oldctrl,
				   ControlData *newctrl)
{
	if (oldctrl->align == 0 || oldctrl->align != newctrl->align)
		pg_log(ctx, PG_FATAL,
			   "old and new pg_controldata alignments are invalid or do not match\n");

	if (oldctrl->blocksz == 0 || oldctrl->blocksz != newctrl->blocksz)
		pg_log(ctx, PG_FATAL,
			   "old and new pg_controldata block sizes are invalid or do not match\n");

	if (oldctrl->largesz == 0 || oldctrl->largesz != newctrl->largesz)
		pg_log(ctx, PG_FATAL,
			   "old and new pg_controldata maximum relation segement sizes are invalid or do not match\n");

	if (oldctrl->walsz == 0 || oldctrl->walsz != newctrl->walsz)
		pg_log(ctx, PG_FATAL,
			   "old and new pg_controldata WAL block sizes are invalid or do not match\n");

	if (oldctrl->walseg == 0 || oldctrl->walseg != newctrl->walseg)
		pg_log(ctx, PG_FATAL,
			   "old and new pg_controldata WAL segment sizes are invalid or do not match\n");

	if (oldctrl->ident == 0 || oldctrl->ident != newctrl->ident)
		pg_log(ctx, PG_FATAL,
			   "old and new pg_controldata maximum identifier lengths are invalid or do not match\n");

	if (oldctrl->index == 0 || oldctrl->index != newctrl->index)
		pg_log(ctx, PG_FATAL,
			   "old and new pg_controldata maximum indexed columns are invalid or do not match\n");

	if (oldctrl->toast == 0 || oldctrl->toast != newctrl->toast)
		pg_log(ctx, PG_FATAL,
			   "old and new pg_controldata maximum TOAST chunk sizes are invalid or do not match\n");

	if (oldctrl->date_is_int != newctrl->date_is_int)
	{
		pg_log(ctx, PG_WARNING,
		"old and new pg_controldata date/time storage types do not match\n");

		/*
		 * This is a common 8.3 -> 8.4 migration problem, so we are more
		 * verboase
		 */
		pg_log(ctx, PG_FATAL,
			   "\nYou will need to rebuild the new server with configure\n"
			   "--disable-integer-datetimes or get server binaries built with those options.\n");
	}
}


/*
 * set_locale_and_encoding()
 *
 * query the database to get the template0 locale
 */
void
set_locale_and_encoding(migratorContext *ctx, Cluster whichCluster)
{
	PGconn	   *conn;
	PGresult   *res;
	int			i_encoding;
	ControlData *ctrl = (whichCluster == CLUSTER_OLD) ?
						&ctx->old.controldata : &ctx->new.controldata;
	int			cluster_version = (whichCluster == CLUSTER_OLD) ?
								ctx->old.pg_version : ctx->new.pg_version;
								
	conn = connectToServer(ctx, "template1", whichCluster);

	/* for pg < 80400, we got the values from pg_controldata */
	if (cluster_version >= 80400)
	{
		int			i_datcollate;
		int			i_datctype;

		res = executeQueryOrDie(ctx, conn,
								"SELECT datcollate, datctype "
								"FROM 	pg_catalog.pg_database "
								"WHERE	datname = 'template0' ");
		assert(PQntuples(res) == 1);
	
		i_datcollate = PQfnumber(res, "datcollate");
		i_datctype = PQfnumber(res, "datctype");
	
		ctrl->lc_collate = pg_strdup(ctx, PQgetvalue(res, 0, i_datcollate));
		ctrl->lc_ctype = pg_strdup(ctx, PQgetvalue(res, 0, i_datctype));
	
		PQclear(res);
	}
	
	res = executeQueryOrDie(ctx, conn,
							"SELECT pg_catalog.pg_encoding_to_char(encoding) "
							"FROM 	pg_catalog.pg_database "
							"WHERE	datname = 'template0' ");
	assert(PQntuples(res) == 1);

	i_encoding = PQfnumber(res, "pg_encoding_to_char");
	ctrl->encoding = pg_strdup(ctx, PQgetvalue(res, 0, i_encoding));

	PQclear(res);
		
	PQfinish(conn);
}


/*
 * check_locale_and_encoding()
 *
 *	locale is not in pg_controldata in 8.4 and later so
 *	we probably had to get via a database query.
 */
void
check_locale_and_encoding(migratorContext *ctx, ControlData *oldctrl,
				   ControlData *newctrl)
{
	if (strcmp(oldctrl->lc_collate, newctrl->lc_collate) != 0)
		pg_log(ctx, PG_FATAL,
			   "old and new cluster lc_collate values do not match\n");
	if (strcmp(oldctrl->lc_ctype, newctrl->lc_ctype) != 0)
		pg_log(ctx, PG_FATAL,
			   "old and new cluster lc_ctype values do not match\n");
	if (strcmp(oldctrl->encoding, newctrl->encoding) != 0)
		pg_log(ctx, PG_FATAL,
			   "old and new cluster encoding values do not match\n");
}
