/*
 *	relfilenode.c
 *
 *	relfilenode functions
 */

#include "pg_migrator.h"

#ifdef EDB_NATIVE_LANG
#include <fcntl.h>
#endif

#include "catalog/pg_class.h"
#include "access/transam.h"


static void transfer_single_new_db(migratorContext *ctx, pageCnvCtx *pageConverter,
					   FileNameMap *maps, int size);
static void transfer_relfile(migratorContext *ctx, pageCnvCtx *pageConverter,
				 const char *fromfile, const char *tofile,
				 const char *oldnspname, const char *oldrelname,
				 const char *newnspname, const char *newrelname);

/*
 * reserve_relfilenodes()
 *
 * Creates empty (rel)files for the respective toast tables of each database in
 * their corresponding physical directories. This will prevent new refilenodes
 * (to be created during restoration of the old schema) to conflict with the old
 * toast table refilenodes and Oids.
 *
 * NOTE: One assumption here is that the server will not select an Oid for which
 * a relfilenode already exists in the database directory.
 *
 * NOTE: assumes that dbs are the old database's dbinfos and datadir is the
 * path to the new data directory.
 */
void
reserve_relfilenodes(migratorContext *ctx)
{
	int			dbnum;

	prep_status(ctx, "Creating placeholder relfiles for toast relations\n");

	for (dbnum = 0; dbnum < ctx->old.dbarr.ndbs; dbnum++)
	{
		int			relnum;
		DbInfo	   *curr_db = dbarr_lookup_db(&ctx->old.dbarr,
										  ctx->new.dbarr.dbs[dbnum].db_name);

		/* now create the placeholder rel files */
		for (relnum = 0; relnum < curr_db->rel_arr.nrels; relnum++)
		{
			int			fd;
			char		path[MAXPGPATH];

			if (curr_db->rel_arr.rels[relnum].toastrelid != 0)
			{
				snprintf(path, sizeof(path), "%s/base/%u/%u", ctx->new.pgdata,
						 ctx->new.dbarr.dbs[dbnum].db_oid,
						 curr_db->rel_arr.rels[relnum].toastrelid);
				pg_log(ctx, PG_REPORT, OVERWRITE_MESSAGE, path);

				if ((fd = open(path, O_WRONLY | O_CREAT, S_IRUSR | S_IWUSR)) < 0)
					pg_log(ctx, PG_FATAL, "\nCannot create file %s\n", path);
				close(fd);
			}
		}
	}

	prep_status(ctx, "");
	check_ok(ctx);
}


/*
 * new_8_4_adjust_rels_to_use_fixed_toast_relfilenodes()
 * new <= 8.4
 * Adjust empty relfiles to use old relfilenodes for TOAST tables
 * because TOAST relfilenodes are stored in user tables.
 */
void
new_8_4_adjust_rels_to_use_fixed_toast_relfilenodes(migratorContext *ctx,
								  DbInfoArr *olddb_arr, DbInfoArr *newdb_arr)
{
	int			dbnum;
	int			relnum;
	DbInfo	   *olddb;

	for (dbnum = 0; dbnum < newdb_arr->ndbs; dbnum++)
	{
		DbInfo	   *newdb = &newdb_arr->dbs[dbnum];
		PGconn	   *conn = connectToServer(ctx, newdb->db_name, CLUSTER_NEW);

		/* find the corresponding old db. Skip the iteration if not found */
		olddb = dbarr_lookup_db(olddb_arr, newdb->db_name);

		/* finally recreate the new toast tables */
		for (relnum = 0; relnum < newdb->rel_arr.nrels; relnum++)
		{
			RelInfo    *newrel = &newdb->rel_arr.rels[relnum];
			RelInfo    *oldrel;
			char		unlink_path[MAXPGPATH];

			/*
			 * New toast relations don't exist in the old cluster, and are
			 * deleted below.
			 */
			if (strcmp(newrel->nspname, "pg_toast") == 0)
				continue;

			oldrel = relarr_lookup_rel(ctx, &olddb->rel_arr,
							  newrel->nspname, newrel->relname, CLUSTER_OLD);

			if (!oldrel->toastrelid)
				continue;

			/*
			 * Unlink the placeholder relfilenodes, including fsm, and vm
			 * There are no extents because the files are empty;  ignore
			 * failure. Nothing references them.  We only delete as we add
			 * each toast table to avoid the risk of having the new toast
			 * table use a placeholder value.
			 */
			snprintf(unlink_path, sizeof(unlink_path), "%s/base/%u/%u",
					 ctx->new.pgdata, newdb->db_oid, oldrel->toastrelid);
			pg_log(ctx, PG_REPORT, OVERWRITE_MESSAGE, unlink_path);
			unlink(unlink_path);
			snprintf(unlink_path, sizeof(unlink_path), "%s/base/%u/%u_fsm",
					 ctx->new.pgdata, newdb->db_oid, oldrel->toastrelid);
			unlink(unlink_path);
			snprintf(unlink_path, sizeof(unlink_path), "%s/base/%u/%u_vm",
					 ctx->new.pgdata, newdb->db_oid, oldrel->toastrelid);
			unlink(unlink_path);

			/* remove pg_class linkage between heap tables and toast tables. */
			PQclear(executeQueryOrDie(ctx, conn,
									  "UPDATE pg_catalog.pg_class "
									  "SET 	reltoastrelid = 0, "
									  "		reltoastidxid = 0 "
									  "WHERE oid = %u ",
									  newrel->reloid));

			/* remove dependency so we can drop toast table */
			PQclear(executeQueryOrDie(ctx, conn,
									  "DELETE FROM pg_catalog.pg_depend "
									  "WHERE classid = %u AND "
									  "objid = %u ",
									  RelationRelationId,
									  newrel->toastrelid));
			/* drop toast table and its index */
			PQclear(executeQueryOrDie(ctx, conn,
								"SELECT binary_upgrade.pg_toasttbl_drop(%u)",
									  newrel->toastrelid));
			/* recreate toast table and its index with original old oids */
			PQclear(executeQueryOrDie(ctx, conn,
						"SELECT binary_upgrade.pg_toasttbl_recreate(%u, %u)",
									  newrel->reloid, oldrel->toastrelid));
		}

		PQfinish(conn);
	}
}


/*
 * transfer_all_new_dbs()
 *
 * Responsible for upgrading all database. invokes routines to generate mappings and then
 * physically link the databases.
 */
const char *
transfer_all_new_dbs(migratorContext *ctx, DbInfoArr *olddb_arr,
					 DbInfoArr *newdb_arr, char *old_pgdata, char *new_pgdata)
{
	int			dbnum;
	const char *msg = NULL;

	prep_status(ctx, "Restoring user relation files\n");

	for (dbnum = 0; dbnum < newdb_arr->ndbs; dbnum++)
	{
		DbInfo	   *new_db = &newdb_arr->dbs[dbnum];
		DbInfo	   *old_db = dbarr_lookup_db(olddb_arr, new_db->db_name);
		FileNameMap *mappings;
		int			n_maps;
		pageCnvCtx *pageConverter = NULL;

		n_maps = 0;
		mappings = gen_db_file_maps(ctx, old_db, new_db, &n_maps, old_pgdata,
									new_pgdata);

		if (n_maps)
		{
			print_maps(ctx, mappings, n_maps, new_db->db_name);

#ifdef PAGE_CONVERSION
			msg = setupPageConverter(ctx, &pageConverter);
#endif
			transfer_single_new_db(ctx, pageConverter, mappings, n_maps);

			pg_free(mappings);
		}
	}

	prep_status(ctx, "");		/* in case nothing printed */
	check_ok(ctx);

	return msg;
}


/*
 * get_pg_database_relfilenode()
 *
 *	Retrieves the relfilenode for a few system-catalog tables.	We need these
 *	relfilenodes later in the upgrade process.
 */
void
get_pg_database_relfilenode(migratorContext *ctx, Cluster whichCluster)
{
	PGconn	   *conn = connectToServer(ctx, "template1", whichCluster);
	PGresult   *res;
	int			i_relfile;

	res = executeQueryOrDie(ctx, conn,
							"SELECT c.relname, c.relfilenode "
							"FROM 	pg_catalog.pg_class c, "
							"		pg_catalog.pg_namespace n "
							"WHERE 	c.relnamespace = n.oid AND "
							"		n.nspname = 'pg_catalog' AND "
							"		c.relname = 'pg_database' "
							"ORDER BY c.relname");

	i_relfile = PQfnumber(res, "relfilenode");
	if (whichCluster == CLUSTER_OLD)
		ctx->old.pg_database_oid = atol(PQgetvalue(res, 0, i_relfile));
	else
		ctx->new.pg_database_oid = atol(PQgetvalue(res, 0, i_relfile));

	PQclear(res);
	PQfinish(conn);
}


/*
 * transfer_single_new_db()
 *
 * create links for mappings stored in "maps" array.
 */
static void
transfer_single_new_db(migratorContext *ctx, pageCnvCtx *pageConverter,
					   FileNameMap *maps, int size)
{
	int			mapnum;

	for (mapnum = 0; mapnum < size; mapnum++)
	{
		char		old_file[MAXPGPATH];
		char		new_file[MAXPGPATH];
		struct dirent **namelist = NULL;
		int			numFiles;

		/* Copying files might take some time, so give feedback. */

		snprintf(old_file, sizeof(old_file), "%s/%u", maps[mapnum].old_file, maps[mapnum].old);
		snprintf(new_file, sizeof(new_file), "%s/%u", maps[mapnum].new_file, maps[mapnum].new);
		pg_log(ctx, PG_REPORT, OVERWRITE_MESSAGE, old_file);

		/*
		 * Copy/link the relation file to the new cluster
		 */
		unlink(new_file);
		transfer_relfile(ctx, pageConverter, old_file, new_file,
						 maps[mapnum].old_nspname, maps[mapnum].old_relname,
						 maps[mapnum].new_nspname, maps[mapnum].new_relname);

		if (GET_MAJOR_VERSION(ctx->old.major_version) >= 804 &&
			GET_MAJOR_VERSION(ctx->new.major_version) >= 804)
		{
			/*
			 * Now copy/link any fsm and vm files, if they exist
			 */
			snprintf(scandir_file_pattern, sizeof(scandir_file_pattern), "%u_", maps[mapnum].old);
			numFiles = pg_scandir(ctx, maps[mapnum].old_file, &namelist, dir_matching_filenames, NULL);

			while (numFiles--)
			{
				snprintf(old_file, sizeof(old_file), "%s/%s", maps[mapnum].old_file,
						 namelist[numFiles]->d_name);
				snprintf(new_file, sizeof(new_file), "%s/%u%s", maps[mapnum].new_file,
				  maps[mapnum].new, strchr(namelist[numFiles]->d_name, '_'));

				unlink(new_file);
				transfer_relfile(ctx, pageConverter, old_file, new_file,
						  maps[mapnum].old_nspname, maps[mapnum].old_relname,
						 maps[mapnum].new_nspname, maps[mapnum].new_relname);

				pg_free(namelist[numFiles]);
			}

			pg_free(namelist);
		}

		/*
		 * Now copy/link any related segments as well. Remember, PG breaks
		 * large files into 1GB segments, the first segment has no extension,
		 * subsequent segments are named relfilenode.1, relfilenode.2,
		 * relfilenode.3, ...  'fsm' and 'vm' files use underscores so are not
		 * copied.
		 */
		snprintf(scandir_file_pattern, sizeof(scandir_file_pattern), "%u.", maps[mapnum].old);
		numFiles = pg_scandir(ctx, maps[mapnum].old_file, &namelist, dir_matching_filenames, NULL);

		while (numFiles--)
		{
			snprintf(old_file, sizeof(old_file), "%s/%s", maps[mapnum].old_file,
					 namelist[numFiles]->d_name);
			snprintf(new_file, sizeof(new_file), "%s/%u%s", maps[mapnum].new_file,
				  maps[mapnum].new, strchr(namelist[numFiles]->d_name, '.'));

			unlink(new_file);
			transfer_relfile(ctx, pageConverter, old_file, new_file,
						  maps[mapnum].old_nspname, maps[mapnum].old_relname,
						 maps[mapnum].new_nspname, maps[mapnum].new_relname);

			pg_free(namelist[numFiles]);
		}

		pg_free(namelist);
	}
}


/*
 * transfer_relfile()
 *
 * Copy or link file from old cluster to new one.
 */
static void
transfer_relfile(migratorContext *ctx, pageCnvCtx *pageConverter, const char *oldfile,
		 const char *newfile, const char *oldnspname, const char *oldrelname,
				 const char *newnspname, const char *newrelname)
{
	const char *msg;

	if ((ctx->transfer_mode == TRANSFER_MODE_LINK) && (pageConverter != NULL))
		pg_log(ctx, PG_FATAL, "this migration requires page-by-page conversion, "
			   "you must use copy-mode instead of link-mode\n");

	if (ctx->transfer_mode == TRANSFER_MODE_COPY)
	{
		pg_log(ctx, PG_INFO, "copying %s to %s\n", oldfile, newfile);

		if ((msg = copyAndUpdateFile(ctx, pageConverter, oldfile, newfile, true)) != NULL)
			pg_log(ctx, PG_FATAL, "error while copying %s.%s(%s) to %s.%s(%s): %s\n",
				   oldnspname, oldrelname, oldfile, newnspname, newrelname, newfile, msg);
	}
	else
	{
		pg_log(ctx, PG_INFO, "linking %s to %s\n", newfile, oldfile);

		if ((msg = linkAndUpdateFile(ctx, pageConverter, oldfile, newfile)) != NULL)
			pg_log(ctx, PG_FATAL,
			   "error while creating link from %s.%s(%s) to %s.%s(%s): %s\n",
				   oldnspname, oldrelname, oldfile, newnspname, newrelname,
				   newfile, msg);
	}
	return;
}
