/*
 *	version.c
 *
 *	Postgres-version-specific routines
 */

#include "pg_migrator.h"

#include "access/transam.h"


/*
 * new_8_4_check_for_composite_types()
 *	new <= 8.4
 *	Composite types have pg_type oids in their data values and
 *	pg_type.oid is not preserved between migrations prior to PG 8.5.
 *	We don't have to worry about arrays of composite types because we
 *	check arrays later.
 */
void
new_8_4_check_for_composite_types(migratorContext *ctx, Cluster whichCluster)
{
	ClusterInfo *active_cluster = (whichCluster == CLUSTER_OLD) ?
	&ctx->old : &ctx->new;
	int			dbnum;
	FILE	   *script = NULL;
	bool		found = false;
	char		output_path[MAXPGPATH];

	prep_status(ctx, "Checking for columns with user-defined composite types");

	snprintf(output_path, sizeof(output_path), "%s/tables_using_composite_types.txt",
			 ctx->output_dir);

	for (dbnum = 0; dbnum < active_cluster->dbarr.ndbs; dbnum++)
	{
		PGresult   *res;
		bool		db_used = false;
		int			ntups;
		int			rowno;
		int			i_nspname,
					i_relname,
					i_attname;
		DbInfo	   *active_db = &active_cluster->dbarr.dbs[dbnum];
		PGconn	   *conn = connectToServer(ctx, active_db->db_name, whichCluster);

		/* Find any user-defined tsquery columns */
		res = executeQueryOrDie(ctx, conn,
								"SELECT n.nspname, c.relname, a.attname "
								"FROM	pg_catalog.pg_class c, "
								"		pg_catalog.pg_namespace n, "
								"		pg_catalog.pg_attribute a, "
								"		pg_catalog.pg_type t "
								"WHERE	c.relkind = 'r' AND "
								"		c.oid = a.attrelid AND "
								"		NOT a.attisdropped AND "
								"		a.atttypid = t.oid AND "
								"		c.relnamespace = n.oid AND "
								"		t.typtype = 'c' AND "
								"		t.oid >= %u AND "
							  "		n.nspname != 'pg_catalog' AND "
						  "		n.nspname != 'information_schema'",
/* oids assigned by initdb might change, so use FirstBootstrapObjectId */
								FirstBootstrapObjectId);

		ntups = PQntuples(res);
		i_nspname = PQfnumber(res, "nspname");
		i_relname = PQfnumber(res, "relname");
		i_attname = PQfnumber(res, "attname");
		for (rowno = 0; rowno < ntups; rowno++)
		{
			found = true;
			if (script == NULL && (script = fopen(output_path, "w")) == NULL)
				pg_log(ctx, PG_FATAL, "Could not create necessary file:  %s\n", output_path);
			if (!db_used)
			{
				fprintf(script, "Database:  %s\n", active_db->db_name);
				db_used = true;
			}
			fprintf(script, "  %s.%s.%s\n",
					PQgetvalue(res, rowno, i_nspname),
					PQgetvalue(res, rowno, i_relname),
					PQgetvalue(res, rowno, i_attname));
		}

		PQclear(res);

		PQfinish(conn);
	}

	if (found)
	{
		fclose(script);
		pg_log(ctx, PG_REPORT, "fatal\n");
		pg_log(ctx, PG_FATAL,
			   "| Your installation uses composite types.\n"
			   "| These types are not supported for upgrade because\n"
			   "| they contain an internal pg_type.oid that cannot be\n"
			   "| migrated.  You can remove the problem columns and\n"
			   "| restart the migration.  A list of the problem columns\n"
			   "| is in the file:\n"
			   "| \t%s\n\n", output_path);
	}
	else
		check_ok(ctx);
}


/*
 * new_8_4_check_for_array_types()
 *	new <= 8.4
 *	array types have pg_type oids in their data values and
 *	pg_type.oid is not preserved between migrations prior to PG 8.5.
 *	We catch arrays of composite types here too.
 */
void
new_8_4_check_for_array_types(migratorContext *ctx, Cluster whichCluster)
{
	ClusterInfo *active_cluster = (whichCluster == CLUSTER_OLD) ?
	&ctx->old : &ctx->new;
	int			dbnum;
	FILE	   *script = NULL;
	bool		found = false;
	char		output_path[MAXPGPATH];

	prep_status(ctx, "Checking for columns with user-defined array types");

	snprintf(output_path, sizeof(output_path), "%s/tables_using_array_types.txt",
			 ctx->output_dir);

	for (dbnum = 0; dbnum < active_cluster->dbarr.ndbs; dbnum++)
	{
		PGresult   *res;
		bool		db_used = false;
		int			ntups;
		int			rowno;
		int			i_nspname,
					i_relname,
					i_attname;
		DbInfo	   *active_db = &active_cluster->dbarr.dbs[dbnum];
		PGconn	   *conn = connectToServer(ctx, active_db->db_name, whichCluster);

		/* Find any user-defined tsquery columns */
		res = executeQueryOrDie(ctx, conn,
								"SELECT n.nspname, c.relname, a.attname "
								"FROM	pg_catalog.pg_class c, "
								"		pg_catalog.pg_namespace n, "
								"		pg_catalog.pg_attribute a, "
								"		pg_catalog.pg_type t "
								"WHERE	c.relkind = 'r' AND "
								"		c.oid = a.attrelid AND "
								"		NOT a.attisdropped AND "
								"		a.atttypid = t.oid AND "
								"		c.relnamespace = n.oid AND "
								"		t.oid = "
								"		( "
								"			SELECT t2.typarray "
								"			FROM pg_type t2 "
						   "			WHERE t2.oid = t.typelem "
								"		) AND "
								"		t.oid >= %u AND "
							  "		n.nspname != 'pg_catalog' AND "
						  "		n.nspname != 'information_schema'",
/* oids assigned by initdb might change, so use FirstBootstrapObjectId */
								FirstBootstrapObjectId);

		ntups = PQntuples(res);
		i_nspname = PQfnumber(res, "nspname");
		i_relname = PQfnumber(res, "relname");
		i_attname = PQfnumber(res, "attname");
		for (rowno = 0; rowno < ntups; rowno++)
		{
			found = true;
			if (script == NULL && (script = fopen(output_path, "w")) == NULL)
				pg_log(ctx, PG_FATAL, "Could not create necessary file:  %s\n", output_path);
			if (!db_used)
			{
				fprintf(script, "Database:  %s\n", active_db->db_name);
				db_used = true;
			}
			fprintf(script, "  %s.%s.%s\n",
					PQgetvalue(res, rowno, i_nspname),
					PQgetvalue(res, rowno, i_relname),
					PQgetvalue(res, rowno, i_attname));
		}

		PQclear(res);

		PQfinish(conn);
	}

	if (found)
	{
		fclose(script);
		pg_log(ctx, PG_REPORT, "fatal\n");
		pg_log(ctx, PG_FATAL,
			   "| Your installation uses array types.\n"
			   "| These types are not supported for upgrade because\n"
			   "| they contain an internal pg_type.oid that cannot be\n"
			   "| migrated.  You can remove the problem columns and\n"
			   "| restart the migration.  A list of the problem columns\n"
			   "| is in the file:\n"
			   "| \t%s\n\n", output_path);
	}
	else
		check_ok(ctx);
}


/*
 * new_8_4_check_for_enum_types()
 *	new <= 8.4
 *	enum types have pg_type oids in their data values and
 *	pg_type.oid is not preserved between migrations prior to PG 8.5.
 */
void
new_8_4_check_for_enum_types(migratorContext *ctx, Cluster whichCluster)
{
	ClusterInfo *active_cluster = (whichCluster == CLUSTER_OLD) ?
	&ctx->old : &ctx->new;
	int			dbnum;
	FILE	   *script = NULL;
	bool		found = false;
	char		output_path[MAXPGPATH];

	prep_status(ctx, "Checking for columns with user-defined enum types");

	snprintf(output_path, sizeof(output_path), "%s/tables_using_enum_types.txt",
			 ctx->output_dir);

	for (dbnum = 0; dbnum < active_cluster->dbarr.ndbs; dbnum++)
	{
		PGresult   *res;
		bool		db_used = false;
		int			ntups;
		int			rowno;
		int			i_nspname,
					i_relname,
					i_attname;
		DbInfo	   *active_db = &active_cluster->dbarr.dbs[dbnum];
		PGconn	   *conn = connectToServer(ctx, active_db->db_name, whichCluster);

		/* Find any user-defined tsquery columns */
		res = executeQueryOrDie(ctx, conn,
								"SELECT n.nspname, c.relname, a.attname "
								"FROM	pg_catalog.pg_class c, "
								"		pg_catalog.pg_namespace n, "
								"		pg_catalog.pg_attribute a, "
								"		pg_catalog.pg_type t "
								"WHERE	c.relkind = 'r' AND "
								"		c.oid = a.attrelid AND "
								"		NOT a.attisdropped AND "
								"		a.atttypid = t.oid AND "
								"		c.relnamespace = n.oid AND "
								"		t.typtype = 'e' AND "
								"		t.oid >= %u AND "
							  "		n.nspname != 'pg_catalog' AND "
						  "		n.nspname != 'information_schema'",
/* oids assigned by initdb might change, so use FirstBootstrapObjectId */
								FirstBootstrapObjectId);

		ntups = PQntuples(res);
		i_nspname = PQfnumber(res, "nspname");
		i_relname = PQfnumber(res, "relname");
		i_attname = PQfnumber(res, "attname");
		for (rowno = 0; rowno < ntups; rowno++)
		{
			found = true;
			if (script == NULL && (script = fopen(output_path, "w")) == NULL)
				pg_log(ctx, PG_FATAL, "Could not create necessary file:  %s\n", output_path);
			if (!db_used)
			{
				fprintf(script, "Database:  %s\n", active_db->db_name);
				db_used = true;
			}
			fprintf(script, "  %s.%s.%s\n",
					PQgetvalue(res, rowno, i_nspname),
					PQgetvalue(res, rowno, i_relname),
					PQgetvalue(res, rowno, i_attname));
		}

		PQclear(res);

		PQfinish(conn);
	}

	if (found)
	{
		fclose(script);
		pg_log(ctx, PG_REPORT, "fatal\n");
		pg_log(ctx, PG_FATAL,
			   "| Your installation uses enum types.\n"
			   "| These types are not supported for upgrade because\n"
			   "| they contain an internal pg_type.oid that cannot be\n"
			   "| migrated.  You can remove the problem columns and\n"
			   "| restart the migration.  A list of the problem columns\n"
			   "| is in the file:\n"
			   "| \t%s\n\n", output_path);
	}
	else
		check_ok(ctx);
}


/*
 * new_8_5_populate_pg_largeobject_metadata()
 *	new >= 8.5
 *	8.5 has a new pg_largeobject permission table
 */
void
new_8_5_populate_pg_largeobject_metadata(migratorContext *ctx, bool check_mode,
										 Cluster whichCluster)
{
	ClusterInfo *active_cluster = (whichCluster == CLUSTER_OLD) ?
	&ctx->old : &ctx->new;
	int			dbnum;
	FILE	   *script = NULL;
	bool		found = false;
	char		output_path[MAXPGPATH];

	prep_status(ctx, "Checking for large objects");

	snprintf(output_path, sizeof(output_path), "%s/pg_largeobject.sql",
			 ctx->output_dir);

	for (dbnum = 0; dbnum < active_cluster->dbarr.ndbs; dbnum++)
	{
		PGresult   *res;
		int			i_count;
		DbInfo	   *active_db = &active_cluster->dbarr.dbs[dbnum];
		PGconn	   *conn = connectToServer(ctx, active_db->db_name, whichCluster);

		/* find if there are any large objects */
		res = executeQueryOrDie(ctx, conn,
								"SELECT count(*) "
								"FROM	pg_catalog.pg_largeobject ");

		i_count = PQfnumber(res, "count");
		if (atoi(PQgetvalue(res, 0, i_count)) != 0)
		{
			found = true;
			if (!check_mode)
			{
				if (script == NULL && (script = fopen(output_path, "w")) == NULL)
					pg_log(ctx, PG_FATAL, "Could not create necessary file:  %s\n", output_path);
				fprintf(script, "\\connect %s\n",
						quote_identifier(ctx, active_db->db_name));
				fprintf(script,
						"SELECT pg_catalog.lo_create(t.loid)\n"
						"FROM (SELECT DISTINCT loid FROM pg_catalog.pg_largeobject) AS t;\n");
			}
		}

		PQclear(res);
		PQfinish(conn);
	}

	if (found)
	{
		if (!check_mode)
			fclose(script);
		report_status(ctx, PG_WARNING, "warning");
		if (check_mode)
			pg_log(ctx, PG_WARNING, "\n"
				   "| Your installation contains large objects.\n"
				   "| The new database has an additional large object\n"
				   "| permission table.  After migration, you will be\n"
				   "| given a command to populate the pg_largeobject\n"
				   "| permission table with default permissions.\n\n");
		else
			pg_log(ctx, PG_WARNING, "\n"
				   "| Your installation contains large objects.\n"
				   "| The new database has an additional large object\n"
				   "| permission table so default permissions must be\n"
				   "| defined for all large objects.  The file:\n"
				   "| \t%s\n"
				   "| when executed by psql by the database super-user\n"
				   "| will define the default permissions.\n\n",
				   output_path);
	}
	else
		check_ok(ctx);
}
