/*
 *	relfilenode.c
 *
 *	relfilenode functions
 */

#include "pg_migrator.h"

#ifdef EDB_NATIVE_LANG
#include <fcntl.h>
#endif

#include <sys/stat.h>

#include "catalog/pg_class.h"
#include "access/transam.h"

/*
 * reserve_relfilenodes()
 *
 * Creates empty (rel)files for the respective toast tables of each database in
 * their corresponding physical directories. This will prevent new refilenodes
 * (to be created during restoration of the old schema) to conflict with the old
 * toast table refilenodes and Oids.
 *
 * NOTE: One assumption here is that the server will not select an Oid for which
 * a relfilenode already exists in the database directory.
 *
 * NOTE: assumes that dbs are the old database's dbinfos and datadir is the
 * path to the new data directory.
 */
void
reserve_relfilenodes(migratorContext *ctx)
{
	int			dbnum;

	for (dbnum = 0; dbnum < ctx->old.dbarr.ndbs; dbnum++)
	{
		int			relnum;
		DbInfo	   *curr_db = dbarr_lookup_db(&ctx->old.dbarr,
						ctx->new.dbarr.dbs[dbnum].db_name);

		/* now create the placeholder rel files */
		for (relnum = 0; relnum < curr_db->rel_arr.nrels; relnum++)
		{
			int	fd;
			char path[MAXPGPATH];

			if (curr_db->rel_arr.rels[relnum].toastrelid != 0)
			{
				snprintf(path, sizeof(path), "%s/base/%u/%u", ctx->new.pgdata,
						 ctx->new.dbarr.dbs[dbnum].db_oid,
						 curr_db->rel_arr.rels[relnum].toastrelid);
				pg_log(ctx, PG_REPORT, OVERWRITE_MESSAGE, path);
	
				if ((fd = open(path, O_WRONLY | O_CREAT, S_IRUSR | S_IWUSR)) < 0)
					pg_log(ctx, PG_FATAL, "\nCannot create file %s\n", path);
				close(fd);
			}
		}
	}
}


/*
 * adjust_rels_to_use_fixed_toast_relfilenodes()
 *
 * the core module which handles toast relations.
 */
void
adjust_rels_to_use_fixed_toast_relfilenodes(migratorContext *ctx,
							DbInfoArr *olddb_arr, DbInfoArr *newdb_arr)
{
	int			dbnum;
	int			relnum;
	DbInfo	   *olddb;

	for (dbnum = 0; dbnum < newdb_arr->ndbs; dbnum++)
	{
		DbInfo	   *newdb = &newdb_arr->dbs[dbnum];
		PGconn	   *conn = connectToServer(ctx, newdb->db_name, CLUSTER_NEW);

		/* find the corresponding old db. Skip the iteration if not found */
		olddb = dbarr_lookup_db(olddb_arr, newdb->db_name);

		/* finally recreate the new toast tables */
		for (relnum = 0; relnum < newdb->rel_arr.nrels; relnum++)
		{
			RelInfo    *newrel = &newdb->rel_arr.rels[relnum];
			RelInfo    *oldrel;
			char		unlink_path[MAXPGPATH];

			/*
			 *	New toast relations don't exist in the old cluster, and
			 *	are deleted below.
			 */
			if (strcmp(newrel->nspname, "pg_toast") == 0)
				continue;

			oldrel = relarr_lookup_rel(ctx, &olddb->rel_arr,
								newrel->nspname, newrel->relname, CLUSTER_OLD);

			if (!oldrel->toastrelid)
				continue;

			/*
			 *	Unlink the placeholder relfilenodes, including fsm, and vm
			 *	There are no extents because the files are empty;  ignore failure.
			 *	Nothing references them.  We only delete as we add each toast table
			 *	to avoid the risk of having the new toast table use a placeholder
			 *	value.
			 */
			snprintf(unlink_path, sizeof(unlink_path), "%s/base/%u/%u", 
					  ctx->new.pgdata, newdb->db_oid, oldrel->toastrelid);
		    pg_log(ctx, PG_REPORT, OVERWRITE_MESSAGE, unlink_path);
			unlink(unlink_path);
			snprintf(unlink_path, sizeof(unlink_path), "%s/base/%u/%u_fsm", 
					  ctx->new.pgdata, newdb->db_oid, oldrel->toastrelid);
			unlink(unlink_path);
			snprintf(unlink_path, sizeof(unlink_path), "%s/base/%u/%u_vm", 
					  ctx->new.pgdata, newdb->db_oid, oldrel->toastrelid);
			unlink(unlink_path);

			/* remove pg_class linkage between heap tables and toast tables. */
			PQclear(executeQueryOrDie(ctx, conn,
									"UPDATE pg_catalog.pg_class "
									"SET 	reltoastrelid = 0, "
									"		reltoastidxid = 0 "
									"WHERE oid = %u ",
									newrel->reloid));

			/* remove dependency so we can drop toast table */
			PQclear(executeQueryOrDie(ctx, conn,
									  "DELETE FROM pg_catalog.pg_depend "
									  "WHERE classid = %u AND "
									  "objid = %u ",
									  RelationRelationId,
									  newrel->toastrelid));
			/* drop toast table and its index */
			PQclear(executeQueryOrDie(ctx, conn,
									  "SELECT binary_upgrade.pg_toasttbl_drop(%u)",
									  newrel->toastrelid));
			/* recreate toast table and its index with original old oids */
			PQclear(executeQueryOrDie(ctx, conn,
									  "SELECT binary_upgrade.pg_toasttbl_recreate(%u, %u)",
									  newrel->reloid, oldrel->toastrelid));
		}

		PQfinish(conn);
	}
}


/*
 * transfer_all_new_dbs()
 *
 * Responsible for upgrading all database. invokes routines to generate mappings and then
 * physically link the databases.
 */
const char *
transfer_all_new_dbs(migratorContext *ctx, DbInfoArr *olddb_arr,
					 DbInfoArr *newdb_arr, char *old_pgdata, char *new_pgdata)
{
	int			dbnum;
	const char *msg = NULL;

	for (dbnum = 0; dbnum < newdb_arr->ndbs; dbnum++)
	{
		DbInfo	   *new_db = &newdb_arr->dbs[dbnum];
		DbInfo	   *old_db = dbarr_lookup_db(olddb_arr, new_db->db_name);
		FileNameMap *mappings;
		int			n_maps;
		pageCnvCtx *pageConverter = NULL;

		n_maps = 0;
		mappings = gen_db_file_maps(ctx, old_db, new_db, &n_maps, old_pgdata,
								   new_pgdata);

		if (n_maps)
		{
			print_maps(ctx, mappings, n_maps, new_db->db_name);

#ifdef PAGE_CONVERSION
			msg = setupPageConverter(ctx, &pageConverter);
#endif
			transfer_single_new_db(ctx, pageConverter, mappings, n_maps);

			pg_free(mappings);
		}
	}

	return msg;
}


/*
 * get_pg_database_relfilenode()
 *
 *	Retrieves the relfilenode for a few system-catalog tables.	We need these
 *	relfilenodes later in the upgrade process.
 */
void
get_pg_database_relfilenode(migratorContext *ctx, Cluster whichCluster)
{
	PGconn	   *conn = connectToServer(ctx, "template1", whichCluster);
	PGresult   *res;
	int			i_relfile;

	res = executeQueryOrDie(ctx, conn,
							"SELECT c.relname, c.relfilenode "
							"FROM 	pg_catalog.pg_class c, "
							"		pg_catalog.pg_namespace n "
							"WHERE 	c.relnamespace = n.oid AND "
							"		n.nspname = 'pg_catalog' AND "
							"		c.relname = 'pg_database' "
							"ORDER BY c.relname");

	i_relfile = PQfnumber(res, "relfilenode");
	if (whichCluster == CLUSTER_OLD)
		ctx->old.pg_database_oid = atol(PQgetvalue(res, 0, i_relfile));
	else
		ctx->new.pg_database_oid = atol(PQgetvalue(res, 0, i_relfile));

	PQclear(res);
	PQfinish(conn);
}


/*
 * getDatabaseOIDs()
 *
 *	Returns a result set that lists the OID and name for each non-system
 *	database defined in the given server.
 */
PGresult *
getDatabaseOIDs(PGconn *conn)
{
	return PQexec(conn,
				  "SELECT oid, datname "
				  "FROM pg_catalog.pg_database "
				  "WHERE datname != 'template0'");
}


/*
 * get_tablespace_paths()
 *
 * Scans pg_tablespace and returns a malloc'ed array of all tablespace
 * paths. Its the caller's responsibility to free the array.
 */
void
get_tablespace_paths(migratorContext *ctx)
{
	PGconn	   *conn = connectToServer(ctx, "template1", CLUSTER_OLD);
	PGresult   *res;
	int			ntups;
	int			tblnum;
	int			i_spclocation;

	res = executeQueryOrDie(ctx, conn,
							"SELECT	spclocation "
							"FROM	pg_catalog.pg_tablespace "
							"WHERE	spcname != 'pg_default' AND "
							"		spcname != 'pg_global'");

	ctx->num_tablespaces = ntups = PQntuples(res);
	ctx->tablespaces = (char **) pg_malloc(ctx, ntups * sizeof(char *));

	i_spclocation = PQfnumber(res, "spclocation");

	for (tblnum = 0; tblnum < ntups; tblnum++)
		ctx->tablespaces[tblnum] = pg_strdup(ctx,
						PQgetvalue(res, tblnum, i_spclocation));

	PQclear(res);

	PQfinish(conn);

	return;
}


/*
 * rename_tablespaces()
 *
 * rename tablespace paths in "tablespaces" array to *.old
 *
 * This will fail if the directory is a mount point, i.e. cannot be renamed
 */
void
rename_tablespaces(migratorContext *ctx)
{
	int			tblnum;
	char		old_path[MAXPGPATH];
	char		new_path[MAXPGPATH];

	prep_status(ctx, "Adding \".old\" suffix to tablespace directory names");

	for (tblnum = 0; tblnum < ctx->num_tablespaces; tblnum++)
	{
		/* rename the old tablespace directory to *.old */
		snprintf(old_path, sizeof(old_path), "%s", ctx->tablespaces[tblnum]);
		snprintf(new_path, sizeof(new_path), "%s.old", ctx->tablespaces[tblnum]);

		pg_log(ctx, PG_INFO, "moving %s to %s\n", old_path, new_path);
		if (pg_mv_file(old_path, new_path) != 0)
		{
#ifdef WIN32
			_dosmaperr(GetLastError());
#endif
			pg_log(ctx, PG_FATAL,
				   "error while renaming old tablespace directory %s to %s: %s\n",
				   old_path, new_path, strerror(errno));
		}
		
		/* create an empty directory for the current tablespace */
		if (mkdir(old_path, 0x755) == -1)
			pg_log(ctx, PG_FATAL,
				   "error while creating tablespace directory %s: %s\n",
				   old_path, strerror(errno));
	}

	check_ok(ctx);
}


/*
 * set_tablespace_subdirectory()
 *
 */
void
set_tablespace_subdirectory(migratorContext *ctx, Cluster whichCluster)
{
	ClusterInfo *cluster = (whichCluster ==  CLUSTER_OLD) ? &ctx->old : &ctx->new;

	/* Did we rename the old tablespace to ".old"? */
	if (GET_MAJOR_VERSION(ctx->new.major_version) <= 804)
	{
		if (whichCluster == CLUSTER_OLD)
			cluster->tablespace_suffix = pg_strdup(ctx, ".old");
		else
			cluster->tablespace_suffix = pg_strdup(ctx, "");
	}
	/* Does this cluster lack a version-specific subdirectory? */
	else if (GET_MAJOR_VERSION(cluster->major_version) <= 804)
			cluster->tablespace_suffix = pg_strdup(ctx, "");
	else
	{
		/* This cluster has a version-specific subdirectory */
		cluster->tablespace_suffix = pg_malloc(ctx, 4 + strlen(cluster->major_version_str) +
							  10 /* OIDCHARS */ + 1);
	
		/* The leading slash is needed to start a new directory. */
		sprintf(cluster->tablespace_suffix, "/PG_%s_%d", cluster->major_version_str,
				cluster->controldata.cat_ver);
	}
}
/*
 * create_script_for_old_cluster_deletion()
 *
 */
char *
create_script_for_old_cluster_deletion(migratorContext *ctx)
{
	FILE		*script = NULL;
	char	   *output_path = pg_malloc(ctx, MAXPGPATH);
	int			tblnum;

	prep_status(ctx, "Creating script to delete old cluster");

	snprintf(output_path, MAXPGPATH, "%s/delete_old_cluster.%s",
			ctx->output_dir, EXEC_EXT);

	if ((script = fopen(output_path, "w")) == NULL)
		pg_log(ctx, PG_FATAL, "Could not create necessary file:  %s\n", output_path);

#ifndef WIN32
	/* add shebang header */
	fprintf(script, "#!/bin/sh\n\n");
#endif

	/* delete old cluster's default tablespace */
	fprintf(script, RMDIR_CMD " %s\n", ctx->old.pgdata);

	/* delete old cluster's alternate tablespaces */
	for (tblnum = 0; tblnum < ctx->num_tablespaces; tblnum++)
	{
		if (GET_MAJOR_VERSION(ctx->old.major_version) <= 804)
		{
			/* delete per-database directories */
			int dbnum;
			
			fprintf(script, "\n");
			for (dbnum = 0; dbnum < ctx->new.dbarr.ndbs; dbnum++)
			{
				fprintf(script, RMDIR_CMD " %s%s/%d\n",
						ctx->tablespaces[tblnum], ctx->old.tablespace_suffix,
						ctx->old.dbarr.dbs[dbnum].db_oid);
			}
		}
		else
			/* delete version-specific subdirectory */
			fprintf(script, RMDIR_CMD " %s%s\n",
					ctx->tablespaces[tblnum], ctx->old.tablespace_suffix);
	}

	fclose(script);

	if (chmod(output_path, S_IRWXU) != 0)
		pg_log(ctx, PG_FATAL, "Could not add execute permission to file:  %s\n", output_path);

	check_ok(ctx);

	return output_path;
}

