#include "connection.h"
#include "statement.h"
#include "protocol.h"
#include "common.h"

/*-----------------------------------------------------------------------------
 * FUNCTION: AllocConnection
 *-----------------------------------------------------------------------------
 */
Connection* AllocConnection(Environment* pEnvironment)
{
	int    i;
	TCHAR* pBuffer;
	Connection* pConnection = (Connection*)malloc(sizeof(Connection));

	if (pConnection)
	{
		memset(pConnection, '\0', sizeof(Connection));
		
		pConnection->dbms = NULL;
		/* let's set protocol version */
		/* now supported only v3.0 */
		pConnection->version = 0x00030000;
		pConnection->state  &= CS_DISCONNECTED;

		/* prepare buffers for communication */
		pConnection->pOutBegin = pConnection->outBuffer;
		pConnection->pOutFirst = pConnection->pOutBegin;
		pConnection->pOutLast  = pConnection->pOutBegin;
		pConnection->pOutEnd   = pConnection->pOutBegin + CONNECTION_OUT_BUFFER_SIZE;
	
		pConnection->pInBegin = pConnection->inBuffer;
		pConnection->pInFirst = pConnection->pInBegin;
		pConnection->pInLast  = pConnection->pInBegin;
		pConnection->pInEnd   = pConnection->pInBegin + CONNECTION_IN_BUFFER_SIZE;

		pConnection->socket = SOCKET_ERROR;

		pBuffer = pConnection->parameters_buffer;
		for (i=0;i<DS_PARAMETERS_NUMBER;i++)
		{
			pConnection->parameters[i] = pBuffer;
			pBuffer += c_stDSParameters[i].unMaxValueLength + 1;
		}

		pConnection->environment  = pEnvironment;
		pConnection->cursor_count = 0;
		pConnection->dbms         = NULL;

		/* add to the environment's list */
		AddItem(&pEnvironment->connections, pConnection);
		/* statements list */
		pConnection->statements.handles = NULL;
		pConnection->TransactionState   = TO_NOT_OPENED;
		
		/* set default attributes */
		pConnection->attributes.connection_dead = SQL_CD_TRUE;
		pConnection->attributes.autocommit = SQL_AUTOCOMMIT_ON;
		pConnection->attributes.unicode = SQL_AA_TRUE;

		InitDiag(&pConnection->diag);
		_INIT_CRITICAL_SECTION(pConnection);
	}
	else
	{
		/* can't alloc */
	}

	return pConnection;
}

/*----------------------------------------------------------------------------
 * FUNCTION: FreeConnection
 *----------------------------------------------------------------------------
 */
SQLRETURN
FreeConnection(Connection* pConnection, SQLUSMALLINT Option)
{
	/* close all statements opened on connection */
        
	FreeList(&pConnection->statements, SQL_HANDLE_STMT, 0);
	free(pConnection->dbms);
	pConnection->dbms = NULL;
	
	if (SQL_DROP == Option)
	{
		ENTER_CRITICAL_SECTION(pConnection);
		RemoveItem(&pConnection->environment->connections, pConnection);
		
		FreeDiag(&pConnection->diag);
		_DELETE_CRITICAL_SECTION(pConnection);
		free(pConnection);
		return SQL_SUCCESS;
	}
	
	return SQL_SUCCESS;
}

/*----------------------------------------------------------------------------
 * FUNCTION: Connect
 *---------------------------------------------------------------------------- */
SQLRETURN Connect(Connection* pConnection)
{
	SQLRETURN nRet;

#ifdef UNICODE
	BYTE pwd[PWD_MAX_LENGTH+1];
	BYTE uid[UID_MAX_LENGTH+1];
#else
	const BYTE* pwd;
	const BYTE* uid;
#endif /* UNICODE */


#ifdef WIN32
	/* make sure socket support is up and running */
	WSADATA wsaData;

	if (WSAStartup(MAKEWORD(1, 1), &wsaData))
		return SQL_ERROR;
	WSASetLastError(0);
#endif /* WIN32 */

	/* default database name is equal to user name */
	if (_T('\0') == *pConnection->parameters[DATABASE_PARAM] && _T('\0') != *pConnection->parameters[UID_PARAM])
		_tcscpy(pConnection->parameters[DATABASE_PARAM], pConnection->parameters[UID_PARAM]);

	/* connecting to backend */
	if (SQL_SUCCESS == sock_connect(pConnection))
	{
    BYTE crypt_pwd1[MD5_PASSWD_LEN + 1];
		BYTE crypt_pwd2[MD5_PASSWD_LEN + 1];
		Message msg;

		/* send startup request */
		if (SQL_SUCCESS == (nRet = SendMessageToBackend(pConnection, MSG_StartupMessage, NULL)))
		{
			do
			{
				/* read messages untill we'll get into READY_FOR_QUERY state */
				msg.message = NULL;
				if (SQL_ERROR == GetMessageFromBackend(pConnection, &msg, NULL))
				{
					SetError(SQL_HANDLE_DBC, pConnection, ERR_DBC_INVALID_LOGIN_DATA, NULL);
					return SQL_ERROR;
				}
				/* translate message */
				switch (msg.type)
				{
					case MSG_ReadyForQuery:
						/* backend state is now Ready For Query, quit with success */
						pConnection->state |= CS_READY;
						nRet = SQL_SUCCESS;
						break;
					case MSG_AuthenticationOk:
						/* connected successfully */
						pConnection->state |= CS_CONNECTED;
						break;
					case MSG_AuthenticationMD5Password:
						/* backend needs md5 password reply */

#ifdef UNICODE
	WideCharToMultiByte(0, 0, pConnection->parameters[PWD_PARAM], wcslen(pConnection->parameters[PWD_PARAM])+1, pwd, sizeof(pwd), NULL, NULL);
	WideCharToMultiByte(0, 0, pConnection->parameters[UID_PARAM], wcslen(pConnection->parameters[UID_PARAM])+1, uid, sizeof(uid), NULL, NULL);
#else
	pwd = pConnection->parameters[PWD_PARAM];
	uid = pConnection->parameters[UID_PARAM];
#endif /* UNICODE */

						if (!EncryptMD5(pwd, uid, strlen(uid), crypt_pwd2) ||
						    !EncryptMD5(crypt_pwd2 + sizeof("md5")-1, (BYTE*)msg.message, 4, crypt_pwd1) ||
						    SQL_ERROR == SendMessageToBackend(pConnection, MSG_PasswordMessage, crypt_pwd1)
						   )
						{
							/* impossible to send or wrong password */
							nRet = SQL_ERROR;
						}
						break;
					case MSG_AuthenticationCleartextPassword:
					case MSG_AuthenticationCryptPassword:
					case MSG_AuthenticationKerberosV4:
					case MSG_AuthenticationKerberosV5:
					case MSG_AuthenticationSCMCredential:
						/* unsupported */
						nRet = SQL_ERROR;
						break;
					case MSG_ErrorResponse:
					case MSG_Unknown:
					default:
						break;
				}
				/* free memory, allocated in 'GetMessageFromBackend' */
				if (msg.message)
					free(msg.message);
				if (SQL_ERROR == nRet)
					break;
			} while (!(pConnection->state & CS_READY));
		}
		/* can't send startup message */
	}
	else
	{
		/* can't connect to requested host */
		nRet = SQL_ERROR;	
	}

	/* prepare clear connection */
	free(pConnection->dbms);
	pConnection->dbms = NULL;

	/* get additional info from server */
	if (SQL_ERROR != nRet)
	{
		Statement* pStatement = AllocStatement(pConnection);
		pConnection->cursor_count = 0;
		if (NULL != pStatement)
		{
			TCHAR*     pStr;
			SQLINTEGER length;
#ifdef UNICODE
			TCHAR client_encoding[] = _T("set client_encoding to 'UTF8'");
#else
			TCHAR* client_encoding = GetText("set client_encoding to '?'", nl_langinfo(CODESET), NULL);
#endif /* UNICODE */
			LEAVE_CRITICAL_SECTION(pConnection);
			/* set appropriate client_encoding */
			if ((SQL_ERROR == PrepareQuery(pStatement, client_encoding, SQL_NTS)) ||
			    (SQL_ERROR == ExecuteStatement(pStatement, TRUE))
			   )
			{
				/* unable to set client_encoding to system locale */
				if ((SQL_ERROR != ResetStatement(pStatement)) &&
				    (SQL_ERROR != PrepareQuery(pStatement, _T("set client_encoding to 'SQL_ASCII'"), SQL_NTS)) &&
			      (SQL_ERROR != ExecuteStatement(pStatement, TRUE))
				   );
			}

			/* get dbms version string */
			if ((SQL_ERROR != ResetStatement(pStatement)) &&
			    (SQL_ERROR != PrepareQuery(pStatement, _T("select version()"), STR_SIZEOF("select version()"))) &&
			    (SQL_ERROR != ExecuteStatement(pStatement, TRUE)) &&
			    (SQL_ERROR != Fetch(pStatement, SQL_FETCH_FIRST, 0)) &&
			    (0 != (length = pStatement->ird->id_records[0].bytes_left)) &&
			    (NULL != (pConnection->dbms = (TCHAR*) malloc((length + CHR_SIZEOF(c_szDBMSVersion)+1)*sizeof(TCHAR)))) &&
			    (SQL_ERROR != GetData(pStatement, 1, SQL_C_TCHAR, (pStr = pConnection->dbms+CHR_SIZEOF(c_szDBMSVersion)), (length+1)*sizeof(TCHAR), &length)) &&
			    (pStr = _tcschr(pStr, _T(' ')) + 1)
			   )
			{
				/* parse version and set it */
				pConnection->backend.version_first  = GetInt(&pStr, _T('.'), &length);
				pConnection->backend.version_second = GetInt(&pStr, _T('.'), &length);
				pConnection->backend.version_third  = GetInt(&pStr, _T(' '), &length);

				pConnection->dbms[_stprintf(pConnection->dbms, _T("%02d.%02d.%04d"), pConnection->backend.version_first, pConnection->backend.version_second, pConnection->backend.version_third)] = _T(' ');
				nRet = SQL_SUCCESS;
			}
			ENTER_CRITICAL_SECTION(pConnection);

#ifndef UNICODE
			free(client_encoding);
#endif /* UNICODE */
			FreeStatement(pStatement, SQL_DROP);
		}
	}

	return nRet;
}

/*----------------------------------------------------------------------------
 * FUNCTION: Disconnect
 *---------------------------------------------------------------------------- */
SQLRETURN
Disconnect(Connection* pConnection)
{
	if (0 != (CS_CONNECTED & pConnection->state))
	{
		/* disconnect from backend */
		SendMessageToBackend(pConnection, MSG_Terminate, NULL);
		sock_close(pConnection);
			/* There is no need to check return values - we will close
			 * connection anyway.
			 */
		pConnection->state &= CS_DISCONNECTED;

		/* close all allocated statements */
		return FreeConnection(pConnection, SQL_CLOSE);
	}
	else
		return SQL_SUCCESS;
}


/*-----------------------------------------------------------------------------
 * FUNCTION: ParseConnectionString
 *-----------------------------------------------------------------------------
 */
SQLRETURN
ParseConnectionString(Connection* pConnection, SQLTCHAR* ConnStr, SQLSMALLINT ConnStrLength)
{
	SQLRETURN nRet;
	SQLRETURN nRetTemp;
	int nKeywordLength;
	int nValueLength;
	ParseState state;
	SQLTCHAR* pKeyword;
	SQLTCHAR* pValue;
	SQLSMALLINT i;

	/* check empty input data */
	if (NULL == ConnStr || 0 == ConnStrLength || _T('\0') == ConnStr[0])
		return SQL_NO_DATA;

	if (SQL_NTS == ConnStrLength)
		ConnStrLength = _tcslen(ConnStr);

	/* andyk ! */
	nRet = SQL_SUCCESS;
	/* read incoming string and fill parameters directly */
	state = PS_KEYWORD_WAITING;
	for (i=ConnStrLength;0<i;i--, ConnStr++)
	{
		switch(*ConnStr)
		{
			case _T(' '):
				/* never changes current state */
				break;
			case _T('{'):
				if (PS_VALUE_WAITING == state)
				{
					pValue = ConnStr+1;
					state = PS_VALUE_CLOSE_WAITING;
					break;
				}
				else
				{
					break;
				}
			case _T('}'):
				switch(state)
				{
					case PS_VALUE_CLOSE_WAITING:
						state = PS_DIVISOR_WAITING;
						/* no break! */
					case PS_DIVISOR_WAITING:
						nValueLength = ( ConnStr - pValue ) - 1;
						break;
				}
				break;
			/* such symbols can be used only within { } */
			case _T('['):
			case _T(']'):
			case _T('('):
			case _T(')'):
			case _T(','):
			case _T('?'):
			case _T('*'):
			case _T('!'):
			case _T('@'):
			case _T('='):
				if (PS_KEYWORD == state)
				{
					nKeywordLength = ConnStr - pKeyword;
					state = PS_VALUE_WAITING;
				}
				break;
			case _T(';'):
				switch(state)
				{
					case PS_VALUE_CLOSE_WAITING:
					case PS_KEYWORD_WAITING:
						break;
					case PS_VALUE_WAITING:
						pValue = ConnStr;
						/* no break! */
					case PS_VALUE:
						nValueLength = ConnStr - pValue;
						/* no break! */
					case PS_DIVISOR_WAITING:
						if (SQL_ERROR == (nRetTemp = GetKeyValue(pConnection, pKeyword, nKeywordLength, pValue, nValueLength)) && nRet != SQL_SUCCESS_WITH_INFO)
							nRet = SQL_SUCCESS_WITH_INFO;
						state = PS_KEYWORD_WAITING;
						break;
				}
				break;
			default:
				switch(state)
				{
					case PS_KEYWORD_WAITING:
						pKeyword = ConnStr;
						state = PS_KEYWORD;
						break;
					case PS_VALUE_WAITING:
						pValue = ConnStr;
						state = PS_VALUE;
				}
		}
	}
	switch(state)
	{
		case PS_VALUE:
			nValueLength = ConnStr - pValue;
			/* no break! */
		case PS_DIVISOR_WAITING:
			if (SQL_ERROR == (nRetTemp = GetKeyValue(pConnection, pKeyword, nKeywordLength, pValue, nValueLength)) && nRet != SQL_SUCCESS_WITH_INFO)
				nRet = SQL_SUCCESS_WITH_INFO;
	}
	if (nRet != SQL_SUCCESS_WITH_INFO)
		nRet = SQL_SUCCESS;

	/* DSN was specified - read missing data from Data Source */
	if (_T('\0') != pConnection->dsn[0])
		ReadFromDS(pConnection->parameters, pConnection->dsn, SQL_NTS);
	
	/* check necessary parameters for connect */
	if (_T('\0') == pConnection->parameters[UID_PARAM][0] ||
			_T('\0') == pConnection->parameters[PORT_PARAM][0] ||
			_T('\0') == pConnection->parameters[SERVER_PARAM][0]
	   )
		return SQL_NEED_DATA;

	/* here possible only SQL_SUCCESS or SQL_SUCCESS_WITH_INFO return values */
	return nRet;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: GetKeyValue
 *
 * DESCRIPTION: 
 *
 * RETURN: SQL_SUCCESS - everything good,
 *         SQL_ERROR - unknown keyword
 *-----------------------------------------------------------------------------
 */
SQLRETURN GetKeyValue(Connection* pConnection,
											TCHAR* Key,   unsigned int KeyLength,
											TCHAR* Value, unsigned int ValueLength)
{
	TCHAR* key;
	TCHAR* value;
	ERR err;
	int length;
	int i;

	if (STR_SIZEOF("DSN") == KeyLength)
	{
		if (0 == _tcsncmp(Key, _T("DSN"), KeyLength) && _T('\0') == pConnection->dsn[0])
		{
			length = MIN(ValueLength, DSN_MAX_LENGTH);
			_tcsncpy(pConnection->dsn, Value, length);
			pConnection->dsn[length] = _T('\0');
			return SQL_SUCCESS;
		}
	}
	else if (STR_SIZEOF("DRIVER") == KeyLength)
	{
		if (0 == _tcsncmp(Key, _T("DRIVER"), KeyLength))
		{
			/* no need of parsing - this is our driver name */
			return SQL_SUCCESS;
		}	
	}

	/* recognize keyword, case-sensitive, and set value if it's empty */
	for (i=FIRST_CONNSTR_PARAM;i<DS_PARAMETERS_NUMBER;i++)
	{
		if (KeyLength == _tcslen(c_stDSParameters[i].szKeyName) &&
		    0 == _tcsncmp(c_stDSParameters[i].szKeyName, Key, KeyLength)
		   )
		{
			if (_T('\0') == pConnection->parameters[i][0])
			{
				length = MIN(c_stDSParameters[i].unMaxValueLength, ValueLength);
				_tcsncpy(pConnection->parameters[i], Value, length);
				pConnection->parameters[i][length] = _T('\0');
				return SQL_SUCCESS;
			}
			else
				break;
		}

		err = (DS_PARAMETERS_NUMBER == i) ? ERR_DBC_UNKNOWN_KEY : ERR_DBC_KEY_ALREADY_SET;
	}

	/* unknown keyword -> set error */
	key = (TCHAR*)malloc((KeyLength + ValueLength + 2) * sizeof(TCHAR)); /* one allocation for two strings */
	value = key + KeyLength + 1;

	_tcsncpy(value, Value, ValueLength);
	_tcsncpy(key,   Key,   KeyLength);

	value[ValueLength] = _T('\0');
	key[KeyLength] = _T('\0');

	SetError(SQL_HANDLE_DBC, pConnection, err, key, value, pConnection->parameters[i], NULL);
	free(key);

	return SQL_ERROR;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: BeginTransaction
 *-----------------------------------------------------------------------------
 */
SQLRETURN
BeginTransaction(Statement* pStatement, SQLSMALLINT Option)
{
	Connection* pConnection = pStatement->connection;

	/* check currently opened transactions */
	if (TO_NOT_OPENED == pConnection->TransactionState)
	{
		if (SQL_ERROR == Stmt_SendMessageToBackend(pConnection, MSG_Query, (Statement*)_T("BEGIN")) ||
		    SQL_ERROR == WaitForBackendReply(pConnection, MSG_ReadyForQuery, pStatement)
	     )
		{
			SetError(SQL_HANDLE_STMT, pStatement, ERR_CANT_BEGIN_TRANSACTION, NULL);
			return SQL_ERROR;
		}
		pConnection->TransactionState = (Option << 1); /* this allow us to determine who was 'THE FIRST' */
	}
	else
		pConnection->TransactionState |= Option;

	pStatement->state |= SS_TRANSACTION_OPENED;
	return SQL_SUCCESS;
}


/*-----------------------------------------------------------------------------
 * FUNCTION: EndTransaction
 *-----------------------------------------------------------------------------
 */
SQLRETURN
EndTransaction(SQLSMALLINT HandleType, SQLHANDLE Handle, SQLSMALLINT CompletionType, SQLSMALLINT Option)
{
	SQLSMALLINT  Mask         = Option * 2;
	const TCHAR* pQuery       = NULL;
	Statement*   pStatement   = NULL;
	Connection*  pConnection  = NULL;
	Environment* pEnvironment = NULL;
	int i                     = 0;
	int j                     = 0;

	/* prepare query text to send */
	switch(CompletionType)
	{
		case SQL_COMMIT:
			pQuery = _T("COMMIT");
			break;
		case SQL_ROLLBACK:
			pQuery = _T("ROLLBACK");
			break;
		default:
			/* Driver Manager is not supposed to let us come here */
			SetError(HandleType, Handle, ERR_INVALID_TRANS_OP_CODE, NULL);
			return SQL_ERROR;
	}

	/* translate handle type */
	switch(HandleType)
	{
		case SQL_HANDLE_STMT:
			/* application can't force us to get here - it is driver's call */
			pStatement  = (Statement*)Handle;
			pConnection = pStatement->connection;
			break;
		case SQL_HANDLE_DBC:
			/* only connection's transaction end */
			pConnection = (Connection*)Handle;
			break;
		case SQL_HANDLE_ENV:
			/* end all transactions in the environment */
			pEnvironment = (Environment*)Handle;
			pConnection  = (0 < pEnvironment->connections.used) ? *pEnvironment->connections.handles : NULL;
			break;
		default:
			/* Driver Manager is not supposed to let us come here, even more - it's impossible! */
			SetError(HandleType, Handle, ERR_INVALID_HANDLE_TYPE, NULL);
			return SQL_ERROR;
	}
	
	/* this WHILE needed when we close Environment's transactions */
	while (NULL != pConnection)
	{	/* -- check option's priority: --
		 * driver can't close transaction if 'THE FIRST' was an appication, but the opposite can take place
		 */
		pConnection->TransactionState |= Option;
		pConnection->TransactionState ^= Option;

		if (0 != (Mask & pConnection->TransactionState))
		{/* transaction should be closed and every statement notified */

			/* successfully close transaction */
			if (SQL_ERROR == Stmt_SendMessageToBackend(pConnection, MSG_Query, (Statement*)pQuery) ||
			    SQL_ERROR == WaitForBackendReply(pConnection, MSG_ReadyForQuery, NULL)
				 )
			{
				SetError(HandleType, Handle, ERR_CANT_CLOSE_TRANSACTION, pQuery, NULL);
				return SQL_ERROR;
			}

			/* notify all child statements */
			for(j=pConnection->statements.used-1;j>=0;j--)
			{
				CloseDeclared(((Statement**)pConnection->statements.handles)[j], _T('p'));
				((Statement**)pConnection->statements.handles)[j]->state ^= SS_TRANSACTION_OPENED;
			}

			pConnection->TransactionState = TO_NOT_OPENED;
		}

		/* get next connection from evironment's list if any */
		pConnection = (NULL != pEnvironment && ++i < pEnvironment->connections.used) ? pEnvironment->connections.handles[i] : NULL;
	}

	return SQL_SUCCESS;
}

/*----------------------------------------------------------------------------
 * FUNCTION: GetConnectionInfo
 *----------------------------------------------------------------------------
 */
SQLRETURN
GetConnectionInfo(Connection* pConnection, SQLUSMALLINT InfoType, SQLPOINTER InfoValue,
                  SQLSMALLINT BufferLength, SQLSMALLINT* StringLength)
{
	SQLRETURN       nRet = SQL_SUCCESS;
	SQLUSMALLINT    uSmallInt;
	SQLUINTEGER     uInteger;
	const SQLTCHAR* pString;

	SQLSMALLINT  ret_type;

	switch(InfoType)
	{
		/* --- Driver Information --- */
		case SQL_MAX_DRIVER_CONNECTIONS:              /* 0 */
			uSmallInt = 0;
			ret_type  = SQL_IS_SMALLINT;
			break;
		case SQL_MAX_CONCURRENT_ACTIVITIES:           /* 1 */
			uSmallInt = 1;
			ret_type  = SQL_IS_SMALLINT;
			break;
		case SQL_DATA_SOURCE_NAME:                    /* 2 */
			pString   = pConnection->dsn;
			uSmallInt = SQL_NTS;
			ret_type  = SQL_IS_POINTER;
			break;
/* -- these information types are implemented by the Driver Manager alone --
		case SQL_DRIVER_HDBC:                            3
		case SQL_DRIVER_HENV:                            4
		case SQL_DRIVER_HSTMT:                           5 */
		case SQL_DRIVER_NAME:                         /* 6 */
			pString   = c_szDriverFileName;
			uSmallInt = CHR_SIZEOF(c_szDriverFileName);
			ret_type  = SQL_IS_POINTER;
			break;
		case SQL_DRIVER_VER:                          /* 7 */
			pString   = c_szDriverVersion;
			uSmallInt = CHR_SIZEOF(c_szDriverVersion);
			ret_type  = SQL_IS_POINTER;
			break;
		case SQL_FETCH_DIRECTION:                     /* 8 */
			uInteger = (_T('Y') == *pConnection->parameters[USEBUFFERING_PARAM]) ? SQL_FD_FETCH_NEXT | SQL_FD_FETCH_FIRST | SQL_FD_FETCH_LAST | SQL_FD_FETCH_PRIOR | SQL_FD_FETCH_ABSOLUTE | SQL_FD_FETCH_RELATIVE /*| SQL_FD_FETCH_BOOKMARK */ : SQL_FD_FETCH_NEXT | SQL_FD_FETCH_LAST;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_ODBC_API_CONFORMANCE:                /* 9 */
			uSmallInt = SQL_OAC_LEVEL2;
			ret_type  = SQL_IS_SMALLINT;
			break;
/* -- implemented solely in the Driver Manager --
		case SQL_ODBC_VER:                              10 */
		case SQL_ROW_UPDATES:                        /* 11 */
			pString   = _T("N");
			uSmallInt = STR_SIZEOF("N");
			ret_type  = SQL_IS_POINTER;
			break;
		case SQL_ODBC_SAG_CLI_CONFORMANCE:           /* 12 */
			uSmallInt = SQL_OSCC_COMPLIANT;
			ret_type  = SQL_IS_SMALLINT;
			break;
		case SQL_SERVER_NAME:                        /* 13 */
			pString   = pConnection->parameters[SERVER_PARAM];
			uSmallInt = SQL_NTS;
			ret_type  = SQL_IS_POINTER;
			break;
		case SQL_SEARCH_PATTERN_ESCAPE:              /* 14 */
			pString   = _T("\\");
			uSmallInt = STR_SIZEOF("\\");
			ret_type  = SQL_IS_POINTER;
			break;
		case SQL_ODBC_SQL_CONFORMANCE:               /* 15 */
			uSmallInt = SQL_OSC_CORE;
			ret_type  = SQL_IS_SMALLINT;
			break;
		case SQL_DATABASE_NAME:                      /* 16 */
			pString   = pConnection->parameters[DATABASE_PARAM];
			uSmallInt = SQL_NTS;
			ret_type  = SQL_IS_POINTER;
			break;
		case SQL_DBMS_NAME:                          /* 17 */
			pString   = pConnection->dbms + CHR_SIZEOF(c_szDBMSVersion);
			uSmallInt = (SQLTCHAR*)_tcschr(pString, _T(' ')) - pString;
			ret_type  = SQL_IS_POINTER;
			break;
		case SQL_DBMS_VER:                           /* 18 */
			pString   = pConnection->dbms;
			uSmallInt = SQL_NTS;
			ret_type  = SQL_IS_POINTER;
			break;
		case SQL_ACCESSIBLE_TABLES:                  /* 19 */
		case SQL_ACCESSIBLE_PROCEDURES:              /* 20 */
			pString   = _T("N");
			uSmallInt = STR_SIZEOF("N");
			ret_type  = SQL_IS_POINTER;
			break;
		case SQL_PROCEDURES:                         /* 21 */
			pString   = _T("Y");
			uSmallInt = STR_SIZEOF("Y");
			ret_type  = SQL_IS_POINTER;
			break;
		case SQL_CONCAT_NULL_BEHAVIOR:               /* 22 */
			uSmallInt = SQL_CB_NON_NULL;
			ret_type  = SQL_IS_SMALLINT;
			break;
		case SQL_CURSOR_COMMIT_BEHAVIOR:             /* 23 */
		case SQL_CURSOR_ROLLBACK_BEHAVIOR:           /* 24 */
			uSmallInt = SQL_CB_PRESERVE;
			ret_type  = SQL_IS_SMALLINT;
			break;
		case SQL_DATA_SOURCE_READ_ONLY:              /* 25 */
			pString   = _T("N");
			uSmallInt = STR_SIZEOF("N");
			ret_type  = SQL_IS_POINTER;
			break;
		case SQL_DEFAULT_TXN_ISOLATION:              /* 26 */
			uInteger = SQL_TXN_READ_COMMITTED;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_EXPRESSIONS_IN_ORDERBY:             /* 27 */
			pString   = _T("Y");
			uSmallInt = STR_SIZEOF("Y");
			ret_type  = SQL_IS_POINTER;
			break;
		case SQL_IDENTIFIER_CASE:                    /* 28 */
			/* non-quoted identifiers default to lowercase */
			uSmallInt = SQL_IC_LOWER;
			ret_type  = SQL_IS_SMALLINT;
			break;
		case SQL_IDENTIFIER_QUOTE_CHAR:              /* 29 */
			pString   = _T("\"");
			uSmallInt = STR_SIZEOF("\"");
			ret_type  = SQL_IS_POINTER;
			break;
		case SQL_MAX_COLUMN_NAME_LEN:                /* 30 */
			uSmallInt = COLUMN_MAX_LENGTH;
			ret_type  = SQL_IS_SMALLINT;
			break;
		case SQL_MAX_CURSOR_NAME_LEN:                /* 31 */
		case SQL_MAX_SCHEMA_NAME_LEN:                /* 32 */
			uSmallInt = SCHEMA_MAX_LENGTH;
			ret_type  = SQL_IS_SMALLINT;
			break;
		case SQL_MAX_PROCEDURE_NAME_LEN:             /* 33 */
			uSmallInt = 0;
			ret_type  = SQL_IS_SMALLINT;
			break;
		case SQL_MAX_CATALOG_NAME_LEN:               /* 34 */
		case SQL_MAX_TABLE_NAME_LEN:                 /* 35 */
			uSmallInt = TABLE_MAX_LENGTH;
			ret_type  = SQL_IS_SMALLINT;
			break;
		case SQL_MULT_RESULT_SETS:                   /* 36 */
		case SQL_MULTIPLE_ACTIVE_TXN:                /* 37 */
		case SQL_OUTER_JOINS:                        /* 38 */
			pString   = _T("Y");
			uSmallInt = STR_SIZEOF("Y");
			ret_type  = SQL_IS_POINTER;
			break;
		case SQL_SCHEMA_TERM:                        /* 39 */
			pString   = _T("schema");
			uSmallInt = STR_SIZEOF("schema");
			ret_type  = SQL_IS_POINTER;
			break;
		case SQL_PROCEDURE_TERM:                     /* 40 */
			pString   = _T("procedure");
			uSmallInt = STR_SIZEOF("procedure");
			ret_type  = SQL_IS_POINTER;
			break;
		case SQL_CATALOG_NAME_SEPARATOR:             /* 41 */
		case SQL_CATALOG_TERM:                       /* 42 */
			pString   = _T("");
			uSmallInt = STR_SIZEOF("");
			ret_type  = SQL_IS_POINTER;
			break;
		case SQL_SCROLL_CONCURRENCY:                 /* 43 */
			uInteger = SQL_SCCO_READ_ONLY;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_SCROLL_OPTIONS:                     /* 44 */
			uInteger = SQL_SO_FORWARD_ONLY | SQL_SO_STATIC;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_TABLE_TERM:                         /* 45 */
			pString   = _T("table");
			uSmallInt = STR_SIZEOF("table");
			ret_type  = SQL_IS_POINTER;
			break;
		case SQL_TXN_CAPABLE:                        /* 46 */
			uSmallInt = SQL_TC_ALL;
			ret_type  = SQL_IS_SMALLINT;
			break;
		case SQL_USER_NAME:                          /* 47 */
			pString   = pConnection->parameters[UID_PARAM];
			uSmallInt = SQL_NTS;
			ret_type  = SQL_IS_POINTER;
			break;
		case SQL_CONVERT_FUNCTIONS:                  /* 48 */
			uInteger = SQL_FN_CVT_CAST; /* SQL_FN_CVT_CONVERT - not avaible in 8.0.3 */
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_NUMERIC_FUNCTIONS:                  /* 49 */
			uInteger = /* mathematical */
				         SQL_FN_NUM_ABS      |
                 SQL_FN_NUM_CEILING  |
			           SQL_FN_NUM_DEGREES  |
			           SQL_FN_NUM_EXP      |
			           SQL_FN_NUM_FLOOR    |
			           SQL_FN_NUM_LOG      |
			           SQL_FN_NUM_LOG10    |
			           SQL_FN_NUM_MOD      |
			           SQL_FN_NUM_PI       |
			           SQL_FN_NUM_POWER    |
			           SQL_FN_NUM_RADIANS  |
			           SQL_FN_NUM_RAND     |
			           SQL_FN_NUM_ROUND    |
			           SQL_FN_NUM_SIGN     |
			           SQL_FN_NUM_SQRT     |
			           SQL_FN_NUM_TRUNCATE |

			           /* trigonometric */
			           SQL_FN_NUM_ACOS  |
			           SQL_FN_NUM_ASIN  |
			           SQL_FN_NUM_ATAN  |
			           SQL_FN_NUM_ATAN2 |
			           SQL_FN_NUM_COS   |
			           SQL_FN_NUM_COT   |
			           SQL_FN_NUM_SIN   |
			           SQL_FN_NUM_TAN;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_STRING_FUNCTIONS:                   /* 50 */
			uInteger = SQL_FN_STR_BIT_LENGTH   |
			           SQL_FN_STR_CHAR_LENGTH  |
								 SQL_FN_STR_OCTET_LENGTH |
								 SQL_FN_STR_POSITION     |
								 SQL_FN_STR_SUBSTRING    |
								 SQL_FN_STR_ASCII        |
								 SQL_FN_STR_LENGTH       |
								 SQL_FN_STR_LTRIM        |
								 SQL_FN_STR_REPEAT       |
								 SQL_FN_STR_REPLACE      |
								 SQL_FN_STR_RTRIM        |
			/* these functions have different names - driver should replace their names during parsing query text 
			 * NOT IMPLEMENTED yet
			 */
			           SQL_FN_STR_LCASE        | /* 'lcase' -> 'lower' */
			           SQL_FN_STR_UCASE        | /* 'ucase' -> 'upper' */
			           SQL_FN_STR_CONCAT       | /* 'concate(str1,str2) -> use 'str1 | str2' instead */
			           SQL_FN_STR_CHAR;          /* 'char' -> 'chr' */
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_SYSTEM_FUNCTIONS:                   /* 51 */
			/* these functions have different names - driver should replace their names during parsing query text 
			 * NOT IMPLEMENTED yet
			 */
			uInteger = SQL_FN_SYS_DBNAME    | /* 'database' -> 'current_database' */
			           SQL_FN_SYS_IFNULL    | /* 'ifnull' -> 'coalesce' */
			           SQL_FN_SYS_USERNAME;   /* 'user' -> 'current_user' */
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_TIMEDATE_FUNCTIONS:                 /* 52 */
			uInteger = SQL_FN_TD_NOW               |
			           SQL_FN_TD_EXTRACT           |
			/* these functions have different names - driver should replace their names during parsing query text 
			 * NOT IMPLEMENTED yet
			 */
			           SQL_FN_TD_CURRENT_DATE      | /* 'current_date()' -> 'current_date' */
			           SQL_FN_TD_CURRENT_TIME      | /* 'current_time()' -> 'current_time' */
			           SQL_FN_TD_CURRENT_TIMESTAMP;  /* 'current_timestamp()' -> 'current_timestamp' */
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_CONVERT_BIGINT:                     /* 53 */
		case SQL_CONVERT_BINARY:                     /* 54 */
		case SQL_CONVERT_BIT:                        /* 55 */
		case SQL_CONVERT_CHAR:                       /* 56 */
		case SQL_CONVERT_DATE:                       /* 57 */
		case SQL_CONVERT_DECIMAL:                    /* 58 */
		case SQL_CONVERT_DOUBLE:                     /* 59 */
		case SQL_CONVERT_FLOAT:                      /* 60 */
		case SQL_CONVERT_INTEGER:                    /* 61 */
		case SQL_CONVERT_LONGVARCHAR:                /* 62 */
		case SQL_CONVERT_NUMERIC:                    /* 63 */
		case SQL_CONVERT_REAL:                       /* 64 */
		case SQL_CONVERT_SMALLINT:                   /* 65 */
		case SQL_CONVERT_TIME:                       /* 66 */
		case SQL_CONVERT_TIMESTAMP:                  /* 67 */
		case SQL_CONVERT_TINYINT:                    /* 68 */
		case SQL_CONVERT_VARBINARY:                  /* 69 */
		case SQL_CONVERT_VARCHAR:                    /* 70 */
		case SQL_CONVERT_LONGVARBINARY:              /* 71 */
		/* andyk - conversion is supported but not using 'convert' function for it */
			uInteger = 0;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_TXN_ISOLATION_OPTION:               /* 72 */
			uInteger = SQL_TXN_READ_UNCOMMITTED | SQL_TXN_READ_COMMITTED | SQL_TXN_REPEATABLE_READ | SQL_TXN_SERIALIZABLE;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_INTEGRITY:                          /* 73 */
			pString   = _T("N");
			uSmallInt = STR_SIZEOF("N");
			ret_type  = SQL_IS_POINTER;
			break;
		case SQL_CORRELATION_NAME:                   /* 74 */
			uSmallInt = SQL_CN_ANY;
			ret_type  = SQL_IS_SMALLINT;
			break;
		case SQL_NON_NULLABLE_COLUMNS:               /* 75 */
			uSmallInt = SQL_NNC_NON_NULL;
			ret_type  = SQL_IS_SMALLINT;
			break;
/* -- these information types are implemented by the Driver Manager alone --
		case SQL_DRIVER_HLIB:                           76 */
		case SQL_DRIVER_ODBC_VER:                    /* 77 */
			pString   = c_szDriverODBCVer;
			uSmallInt = CHR_SIZEOF(c_szDriverODBCVer);
			ret_type  = SQL_IS_POINTER;
			break;
		case SQL_LOCK_TYPES:                         /* 78 */
			uInteger = SQL_LCK_NO_CHANGE | SQL_LCK_EXCLUSIVE | SQL_LCK_UNLOCK;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_POS_OPERATIONS:                     /* 79 */
			uInteger = 0;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_POSITIONED_STATEMENTS:              /* 80 */
			uInteger = 0;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_GETDATA_EXTENSIONS:                 /* 81 */
			/* relaxed restrictions for SQLGetData */
			uInteger = SQL_GD_ANY_COLUMN | SQL_GD_ANY_ORDER | SQL_GD_BOUND;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_BOOKMARK_PERSISTENCE:               /* 82 */
			uInteger = 0;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_STATIC_SENSITIVITY:                 /* 83 */
			uInteger = 0;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_FILE_USAGE:                         /* 84 */
			uSmallInt = SQL_FILE_NOT_SUPPORTED;
			ret_type = SQL_IS_SMALLINT;
			break;
		case SQL_NULL_COLLATION:                     /* 85 */
			uSmallInt = SQL_NC_END;
			ret_type = SQL_IS_SMALLINT;
			break;
		case SQL_ALTER_TABLE:                        /* 86 */
			uInteger = SQL_AT_ADD_COLUMN_COLLATION | SQL_AT_ADD_COLUMN_DEFAULT | SQL_AT_ADD_COLUMN_SINGLE | SQL_AT_ADD_CONSTRAINT | SQL_AT_ADD_TABLE_CONSTRAINT | SQL_AT_CONSTRAINT_NAME_DEFINITION | SQL_AT_DROP_COLUMN_CASCADE | SQL_AT_DROP_COLUMN_DEFAULT | SQL_AT_DROP_COLUMN_RESTRICT | SQL_AT_DROP_TABLE_CONSTRAINT_CASCADE | SQL_AT_DROP_TABLE_CONSTRAINT_RESTRICT | SQL_AT_SET_COLUMN_DEFAULT | SQL_AT_CONSTRAINT_INITIALLY_DEFERRED | SQL_AT_CONSTRAINT_INITIALLY_IMMEDIATE | SQL_AT_CONSTRAINT_DEFERRABLE;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_COLUMN_ALIAS:                       /* 87 */
			pString   = _T("Y");
			uSmallInt = STR_SIZEOF("Y");
			ret_type  = SQL_IS_POINTER;
			break;
		case SQL_GROUP_BY:                           /* 88 */
			uSmallInt = SQL_GB_GROUP_BY_EQUALS_SELECT;
			ret_type = SQL_IS_SMALLINT;
			break;
		case SQL_KEYWORDS:                           /* 89 */
			pString   = c_szKeyWords;
			uSmallInt = SQL_NTS;
			ret_type  = SQL_IS_POINTER;
			break;
		case SQL_ORDER_BY_COLUMNS_IN_SELECT:         /* 90 */
			pString   = _T("N");
			uSmallInt = STR_SIZEOF("N");
			ret_type  = SQL_IS_POINTER;
			break;
		case SQL_OWNER_USAGE:                        /* 91 */
			uInteger = SQL_SU_DML_STATEMENTS | SQL_SU_TABLE_DEFINITION | SQL_SU_INDEX_DEFINITION | SQL_SU_PRIVILEGE_DEFINITION;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_QUALIFIER_USAGE:                    /* 92 */
			uInteger = 0;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_QUOTED_IDENTIFIER_CASE:             /* 93 */
			uSmallInt = SQL_IC_SENSITIVE;
			ret_type = SQL_IS_SMALLINT;
			break;
		case SQL_SPECIAL_CHARACTERS:                 /* 94 */
			pString   = c_szSpecialCharacter;
			uSmallInt = SQL_NTS;
			ret_type  = SQL_IS_POINTER;
			break;
		case SQL_SUBQUERIES:                         /* 95 */
			uInteger = SQL_SQ_CORRELATED_SUBQUERIES | SQL_SQ_QUANTIFIED | SQL_SQ_EXISTS | SQL_SQ_IN | SQL_SQ_COMPARISON;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_UNION:                              /* 96 */
			uInteger = SQL_U_UNION | SQL_U_UNION_ALL;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_MAX_COLUMNS_IN_GROUP_BY:            /* 97 */
		case SQL_MAX_COLUMNS_IN_INDEX:               /* 98 */
		case SQL_MAX_COLUMNS_IN_ORDER_BY:            /* 99 */ /* information_schema.sql_sizing - sizing_id = 99 */
			uSmallInt = 0;
			ret_type  = SQL_IS_SMALLINT;
			break;
		case SQL_MAX_COLUMNS_IN_SELECT:             /* 100 */ /* information_schema.sql_sizing - sizing_id = 100 */
			uSmallInt = 1664;
			ret_type  = SQL_IS_SMALLINT;
			break;
		case SQL_MAX_COLUMNS_IN_TABLE:              /* 101 */ /* information_schema.sql_sizing - sizing_id = 101 */
			uSmallInt = 1600;
			ret_type  = SQL_IS_SMALLINT;
			break;
		case SQL_MAX_INDEX_SIZE:                    /* 102 */
			uInteger = 0;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_MAX_ROW_SIZE_INCLUDES_LONG:        /* 103 */
			pString   = _T("N");
			uSmallInt = STR_SIZEOF("N");
			ret_type  = SQL_IS_POINTER;
			break;
		case SQL_MAX_ROW_SIZE:                      /* 104 */
			uInteger = 0;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_MAX_STATEMENT_LEN:                 /* 105 */
			uInteger = 0;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_MAX_TABLES_IN_SELECT:              /* 106 */  /* information_schema.sql_sizing - sizing_id = 106 */
			uSmallInt = 0;
			ret_type  = SQL_IS_SMALLINT;			
			break;
		case SQL_MAX_USER_NAME_LEN:                 /* 107 */  /* information_schema.sql_sizing - sizing_id = 107 */
			uSmallInt = 63;
			ret_type  = SQL_IS_SMALLINT;
			break;
		case SQL_MAX_CHAR_LITERAL_LEN:              /* 108 */
			uInteger = 0;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_TIMEDATE_ADD_INTERVALS:            /* 109 */
		case SQL_TIMEDATE_DIFF_INTERVALS:           /* 110 */
			uInteger = SQL_FN_TSI_FRAC_SECOND |
			           SQL_FN_TSI_SECOND      |
								 SQL_FN_TSI_MINUTE      |
								 SQL_FN_TSI_HOUR        |
								 SQL_FN_TSI_DAY         |
								 SQL_FN_TSI_WEEK        |
                 SQL_FN_TSI_MONTH       |
                 SQL_FN_TSI_QUARTER     |
                 SQL_FN_TSI_YEAR;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_NEED_LONG_DATA_LEN:                /* 111 */
			pString   = _T("Y");
			uSmallInt = STR_SIZEOF("Y");
			ret_type  = SQL_IS_POINTER;
			break;
		case SQL_MAX_BINARY_LITERAL_LEN:            /* 112 */
			uInteger = 0;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_LIKE_ESCAPE_CLAUSE:                /* 113 */
			pString   = _T("N");
			uSmallInt = STR_SIZEOF("N");
			ret_type  = SQL_IS_POINTER;
			break;
		case SQL_CATALOG_LOCATION:                  /* 114 */
			uSmallInt = SQL_CL_START;
			ret_type  = SQL_IS_SMALLINT;
			break;
		case SQL_OJ_CAPABILITIES:                   /* 115 */
			uInteger = SQL_OJ_LEFT               |
			           SQL_OJ_RIGHT              |
								 SQL_OJ_FULL               |
			           SQL_OJ_NOT_ORDERED        |
			           SQL_OJ_ALL_COMPARISON_OPS;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_ACTIVE_ENVIRONMENTS:               /* 116 */
			/* Maximum number of active environments that the driver can support */
			uSmallInt = 0;
			ret_type  = SQL_IS_SMALLINT;
			break;
		case SQL_ALTER_DOMAIN:                      /* 117 */
			uInteger = SQL_AD_ADD_DOMAIN_CONSTRAINT | SQL_AD_ADD_DOMAIN_DEFAULT | SQL_AD_CONSTRAINT_NAME_DEFINITION | SQL_AD_DROP_DOMAIN_CONSTRAINT | SQL_AD_DROP_DOMAIN_DEFAULT | SQL_AD_ADD_CONSTRAINT_DEFERRABLE | SQL_AD_ADD_CONSTRAINT_NON_DEFERRABLE | SQL_AD_ADD_CONSTRAINT_INITIALLY_DEFERRED | SQL_AD_ADD_CONSTRAINT_INITIALLY_IMMEDIATE;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_SQL_CONFORMANCE:                   /* 118 */
			uInteger = SQL_SC_SQL92_FULL;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_DATETIME_LITERALS:                 /* 119 */
			uInteger = SQL_DL_SQL92_DATE      |
			           SQL_DL_SQL92_TIME      |
			           SQL_DL_SQL92_TIMESTAMP;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_BATCH_ROW_COUNT:                   /* 120 */
			uInteger = SQL_BRC_EXPLICIT;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_BATCH_SUPPORT:                     /* 121 */
			uInteger = SQL_BS_SELECT_EXPLICIT | SQL_BS_ROW_COUNT_EXPLICIT;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_CONVERT_WCHAR:                     /* 122 */
		case SQL_CONVERT_INTERVAL_DAY_TIME:         /* 123 */
		case SQL_CONVERT_INTERVAL_YEAR_MONTH:       /* 124 */
		case SQL_CONVERT_WLONGVARCHAR:              /* 125 */
		case SQL_CONVERT_WVARCHAR:                  /* 126 */
			uInteger = 0;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_CREATE_ASSERTION:                  /* 127 */
		case SQL_CREATE_CHARACTER_SET:              /* 128 */
		case SQL_CREATE_COLLATION:                  /* 129 */
			uInteger = 0;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_CREATE_DOMAIN:                     /* 130 */
			uInteger = 0;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_CREATE_SCHEMA:                     /* 131 */
			uInteger = SQL_CS_CREATE_SCHEMA | SQL_CS_AUTHORIZATION;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_CREATE_TABLE:                      /* 132 */
			uInteger = SQL_CT_CREATE_TABLE | SQL_CT_COLUMN_CONSTRAINT | SQL_CT_COLUMN_DEFAULT | SQL_CT_GLOBAL_TEMPORARY | SQL_CT_TABLE_CONSTRAINT | SQL_CT_CONSTRAINT_NAME_DEFINITION | SQL_CT_CONSTRAINT_INITIALLY_DEFERRED | SQL_CT_CONSTRAINT_INITIALLY_IMMEDIATE | SQL_CT_CONSTRAINT_DEFERRABLE;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_CREATE_TRANSLATION:                /* 133 */
			uInteger = 0;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_CREATE_VIEW:                       /* 134 */
			uInteger = SQL_CV_CREATE_VIEW;
			ret_type = SQL_IS_INTEGER;
			break;
/* -- implemented solely in the Driver Manager --
		case SQL_DRIVER_HDESC:                         135 */
		case SQL_DROP_ASSERTION:                    /* 136 */
		case SQL_DROP_CHARACTER_SET:                /* 137 */
		case SQL_DROP_COLLATION:                    /* 138 */
			uInteger = 0;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_DROP_DOMAIN:                       /* 139 */
			uInteger = SQL_DD_DROP_DOMAIN | SQL_DD_CASCADE | SQL_DD_RESTRICT;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_DROP_SCHEMA:                       /* 140 */
			uInteger = SQL_DS_DROP_SCHEMA | SQL_DS_CASCADE | SQL_DS_RESTRICT;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_DROP_TABLE:                        /* 141 */
			uInteger = SQL_DT_DROP_TABLE | SQL_DT_CASCADE | SQL_DT_RESTRICT;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_DROP_TRANSLATION:                  /* 142 */
			uInteger = 0;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_DROP_VIEW:                         /* 143 */
			uInteger = SQL_DV_DROP_VIEW | SQL_DV_CASCADE | SQL_DV_RESTRICT;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_DYNAMIC_CURSOR_ATTRIBUTES1:        /* 144 */
			uInteger = SQL_CA1_NEXT;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_DYNAMIC_CURSOR_ATTRIBUTES2:        /* 145 */
			uInteger = 0;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_FORWARD_ONLY_CURSOR_ATTRIBUTES1:   /* 146 */
			uInteger = 0;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_FORWARD_ONLY_CURSOR_ATTRIBUTES2:   /* 147 */
			uInteger = SQL_CA2_READ_ONLY_CONCURRENCY;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_INDEX_KEYWORDS:                    /* 148 */
			uInteger = SQL_IK_NONE;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_INFO_SCHEMA_VIEWS:                 /* 149 */
		case SQL_KEYSET_CURSOR_ATTRIBUTES1:         /* 150 */
		case SQL_KEYSET_CURSOR_ATTRIBUTES2:         /* 151 */
			uInteger = 0;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_ODBC_INTERFACE_CONFORMANCE:        /* 152 */
			uInteger = SQL_OIC_CORE;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_PARAM_ARRAY_ROW_COUNTS:            /* 153 */
		case SQL_PARAM_ARRAY_SELECTS:               /* 154 */
			uInteger = SQL_PARC_NO_BATCH;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_SQL92_DATETIME_FUNCTIONS:			    /* 155 */
			uInteger = SQL_SDF_CURRENT_DATE |
			           SQL_SDF_CURRENT_TIME |
			           SQL_SDF_CURRENT_TIMESTAMP;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_SQL92_FOREIGN_KEY_DELETE_RULE:     /* 156 */
		case SQL_SQL92_FOREIGN_KEY_UPDATE_RULE:     /* 157 */
			uInteger = 0;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_SQL92_GRANT:                       /* 158 */
			uInteger = SQL_SG_DELETE_TABLE      |
			           SQL_SG_INSERT_TABLE      |
								 SQL_SG_REFERENCES_TABLE  |
			           SQL_SG_SELECT_TABLE      |
			           SQL_SG_UPDATE_TABLE      |
			           SQL_SG_WITH_GRANT_OPTION;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_SQL92_NUMERIC_VALUE_FUNCTIONS:     /* 159 */
		case SQL_SQL92_PREDICATES:                  /* 160 */
		case SQL_SQL92_RELATIONAL_JOIN_OPERATORS:   /* 161 */
		case SQL_SQL92_REVOKE:                      /* 162 */
		case SQL_SQL92_ROW_VALUE_CONSTRUCTOR:       /* 163 */
		case SQL_SQL92_STRING_FUNCTIONS:            /* 164 */
		case SQL_SQL92_VALUE_EXPRESSIONS:           /* 165 */
			uInteger = 0;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_STANDARD_CLI_CONFORMANCE:          /* 166 */
			uInteger = SQL_SCC_ISO92_CLI;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_STATIC_CURSOR_ATTRIBUTES1:         /* 167 */
		case SQL_STATIC_CURSOR_ATTRIBUTES2:         /* 168 */
			uInteger = 0;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_AGGREGATE_FUNCTIONS:               /* 169 */
			uInteger = SQL_AF_ALL | SQL_AF_AVG | SQL_AF_COUNT | SQL_AF_DISTINCT | SQL_AF_MAX | SQL_AF_MIN | SQL_AF_SUM;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_DDL_INDEX:                         /* 170 */
			uInteger = SQL_DI_CREATE_INDEX | SQL_DI_DROP_INDEX;
			ret_type = SQL_IS_INTEGER;
			break;
/* -- implemented solely in the Driver Manager --
		case SQL_DM_VER:                               171 */
		case SQL_INSERT_STATEMENT:                  /* 172 */
			uInteger = SQL_IS_INSERT_LITERALS | SQL_IS_INSERT_SEARCHED | SQL_IS_SELECT_INTO;
			ret_type = SQL_IS_INTEGER;
			break;
/* unknown param in VC6... */
#ifndef SQL_CONVERT_GUID
	#define SQL_CONVERT_GUID           173
#endif /* SQL_CONVERT_GUID */

		case SQL_CONVERT_GUID:                      /* 173 */
			uInteger = 0;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_XOPEN_CLI_YEAR:                  /* 10000 */
			pString   = _T("1800");
			uSmallInt = STR_SIZEOF("1800");
			ret_type  = SQL_IS_POINTER;
			break;
		case SQL_CURSOR_SENSITIVITY:              /* 10001 */
			uInteger = SQL_UNSPECIFIED;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_DESCRIBE_PARAMETER:              /* 10002 */
			pString   = _T("Y");
			uSmallInt = STR_SIZEOF("Y");
			ret_type  = SQL_IS_POINTER;
			break;
		case SQL_CATALOG_NAME:                    /* 10003 */
			pString   = _T("Y");
			uSmallInt = STR_SIZEOF("Y");
			ret_type  = SQL_IS_POINTER;
			break;
		case SQL_COLLATION_SEQ:                   /* 10004 */
#ifdef UNICODE
			pString   = _T("UNICODE");
			uSmallInt = STR_SIZEOF("UNICODE");
#else
			pString   = _T("");
			uSmallInt = STR_SIZEOF("");
#endif /* UNICODE */
			ret_type  = SQL_IS_POINTER;
			break;
		case SQL_MAX_IDENTIFIER_LEN:              /* 10005 */
			uSmallInt = 63;
			ret_type  = SQL_IS_SMALLINT;
			break;
		case SQL_ASYNC_MODE:                      /* 10021 */
			uInteger = SQL_AM_NONE;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_MAX_ASYNC_CONCURRENT_STATEMENTS: /* 10022 */
			uInteger = 0;
			ret_type = SQL_IS_INTEGER;
			break;

		/* andyk case SQL_ODBC_STANDARD_CLI_CONFORMANCE: */
		/* currently unsupported */
		default:
			break;
	}

	/* return values */
	switch(ret_type)
	{
		case SQL_IS_SMALLINT:
			memcpy(InfoValue, &uSmallInt, sizeof(SQLSMALLINT));
			if (NULL != StringLength)
				*StringLength = sizeof(SQLSMALLINT);
			break;
		case SQL_IS_INTEGER:
			memcpy(InfoValue, &uInteger, sizeof(SQLINTEGER));
			if (NULL != StringLength)
				*StringLength = sizeof(SQLINTEGER);
			break;
		case SQL_IS_POINTER:
			return ReturnString(InfoValue, BufferLength, StringLength, pString, uSmallInt);
	}
	return SQL_SUCCESS;
}

/*----------------------------------------------------------------------------
 * FUNCTION: SetConnectAttr
 *----------------------------------------------------------------------------
 */
SQLRETURN
SetConnectAttr(Connection* pConnection, SQLINTEGER Attribute, SQLPOINTER Value, SQLINTEGER StringLength)
{
	SQLRETURN nRet = SQL_SUCCESS;

	switch(Attribute)
	{
		case SQL_ATTR_LOGIN_TIMEOUT:       /* 103 */
			/* *(SQLUINTEGER*)Value = 0; */
			break;
		case SQL_ATTR_TXN_ISOLATION:       /* 108 */
			/* This attribute can be set only if there is no transaction opened */
			break;
		case SQL_ATTR_CURRENT_CATALOG:     /* 109 */
			SetError(SQL_HANDLE_DBC, pConnection, ERR_CONN_USE_UNSUPPORTED, NULL);
			nRet = SQL_ERROR;
			break;
		case SQL_ATTR_ANSI_APP:            /* 115 */
			pConnection->attributes.unicode = (SQLUINTEGER)Value;
			break;

		/* read only attributes */
		case SQL_ATTR_CONNECTION_DEAD:    /* 1209 */
		case SQL_ATTR_AUTO_IPD:          /* 10001 */
			SetError(SQL_HANDLE_DBC, pConnection, ERR_READONLY_ATTRIBUTE, NULL);
			nRet = SQL_ERROR;
			break;
		case SQL_ATTR_AUTOCOMMIT:          /* 102 */
			pConnection->attributes.autocommit = (SQLUINTEGER)Value;
			break;
		case SQL_ATTR_ACCESS_MODE:         /* 101 */
		case SQL_ATTR_ASYNC_ENABLE:        /*   4 */
		case SQL_ATTR_CONNECTION_TIMEOUT:  /* 113 */
		case SQL_ATTR_METADATA_ID:       /* 10014 */
		case SQL_ATTR_ODBC_CURSORS:        /* 110 */
		case SQL_ATTR_PACKET_SIZE:         /* 112 */
		case SQL_ATTR_QUIET_MODE:          /* 111 */
		case SQL_ATTR_TRACE:               /* 104 */
		case SQL_ATTR_TRACEFILE:           /* 105 */
		case SQL_ATTR_TRANSLATE_LIB:       /* 106 */
		case SQL_ATTR_TRANSLATE_OPTION:    /* 107 */
		default:
			SetError(SQL_HANDLE_DBC, pConnection, ERR_UNKNOWN_ATTRIBUTE, NULL);
			nRet = SQL_ERROR;
	}
	return nRet;
}
