﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using FirebirdSql.Data.FirebirdClient;
using Npgsql;
using NpgsqlTypes;

namespace fb2pg
{
    /// <summary>
    /// Esta clase gestiona la ventana principal de la aplicación.
    /// Aquí están los métodos generales que realizan el traspaso de una base de datos a la otra.
    /// </summary>
    public partial class FormPrincipal : Form
    {
        #region · Campos ·

        /// <summary>
        /// Enumeración que define las columnas que contiene el <see cref="DataGridView"/>
        /// </summary>
        enum ColumnasDgv
        {
            Nombre,
            Tabla,
            Datos,
            Indice,
            Tiempo
        }

        /// <summary>
        /// Conexión con el servidor Firebird
        /// </summary>
        FbConnection connFirebird;
        /// <summary>
        /// Conexión con el servidor PostgreSQL
        /// </summary>
        NpgsqlConnection connPostgres;

        /// <summary>
        /// Lista de generadores de secuencias extraídos de Firebird
        /// </summary>
        List<TablasGeneradores> listaGen;
        /// <summary>
        /// Lista de las claves primarias de las distintas tablas de la base de datos
        /// </summary>
        List<TablasClavesPrimarias> listaPK;
        /// <summary>
        /// Lista de índices que existen en la base de datos
        /// </summary>
        List<TablasIndices> listaIndices;

        /// <summary>
        /// Tiempo total de proceso
        /// </summary>
        TimeSpan tiempoProceso;

        /// <summary>
        /// Número de tablas que contiene la base de datos
        /// </summary>
        int numTablasDgv;

        #endregion

        #region · Constructor ·

        /// <summary>
        /// Crea una instancia de la clase <see cref="FormPrincipal"/> que gestiona la ventana y realiza el traspaso
        /// </summary>
        public FormPrincipal()
        {
            InitializeComponent();


            this.listaGen = new List<TablasGeneradores>();
            this.listaPK = new List<TablasClavesPrimarias>();
            this.listaIndices = new List<TablasIndices>();
        }

        #endregion

        #region · Control de eventos ·

        /// <summary>
        /// Gestiona el botón que utilizamos para iniciar el proceso
        /// </summary>
        /// <param name="sender">Botón que genera el evento</param>
        /// <param name="e">Argumentos del evento</param>
        private void toolStripButton1_Click(object sender, EventArgs e)
        {
            if (this.CreaConexiones())
            {
                this.tiempoProceso = TimeSpan.Zero;

                this.TraspasaBaseDatos();

                this.connFirebird.Close();
                this.connPostgres.Close();
                this.connFirebird = null;
                this.connPostgres = null;
            }
        }

        #endregion

        #region · Métodos de traspaso de la base de datos ·

        /// <summary>
        /// Traspasa la base de datos desde Firebird a PostgreSQL
        /// </summary>
        private void TraspasaBaseDatos()
        {
            DateTime tInicio;


            TablasIndices.RellenaListaTablasIndices(connFirebird, listaIndices);
            TablasGeneradores.RellenaListaTablasGeneradores(connFirebird, listaGen);
            TablasClavesPrimarias.RellenaListaClavesPrimarias(connFirebird, listaPK);

            DataTable dt = connFirebird.GetSchema("Tables");
            //
            // Inicializamos el DataGridViewEdit con los nombres de las tablas
            //
            this.RellenaDataGridView(dt);

            this.toolStripProgressBar1.Visible = true;

            int numFilas = dt.Rows.Count;

            for (int i = 0; i < numFilas; i++)
            {
                DataRow r = dt.Rows[i];
                if (Convert.ToInt32(r["IS_SYSTEM_TABLE"].ToString()) == 0)
                {
                    tInicio = DateTime.Now;
                    string nombreTabla = r["TABLE_NAME"].ToString().ToLowerInvariant();

                    //
                    // Rellenamos una lista de objetos InfoEstructuraFirebird que contiene información de los campos de la tabla
                    //
                    List<InfoEstructuraFirebird> listaCampos = new List<InfoEstructuraFirebird>();
                    InfoEstructuraFirebird.RellenaLista(connFirebird, listaCampos, nombreTabla);

                    this.dataGridView1.CurrentCell = this.dataGridView1[(int)ColumnasDgv.Tabla, i];
                    this.dataGridView1.RefreshEdit();

                    if (!this.CreaTabla(nombreTabla, listaCampos))
                        break;
                    this.ColumnaOK();

                    this.TraspasaDatosTabla(nombreTabla, listaCampos);
                    this.ColumnaOK();

                    this.CreaIndices(nombreTabla);
                    this.ColumnaOK();

                    TimeSpan tiempoProcesoTabla = DateTime.Now - tInicio;
                    this.tiempoProceso += tiempoProcesoTabla;

                    this.dataGridView1.Rows[i].Cells[(int)ColumnasDgv.Tiempo].Value = tiempoProcesoTabla;

                    this.toolStripLabel1.Text = this.tiempoProceso.ToString();

                    Application.DoEvents();
                }
            }


            this.toolStripLabel1.Text = "Secuencias";
            Application.DoEvents();

            tInicio = DateTime.Now;
            this.ActualizaValoresSecuencias();
            TimeSpan tiempoActualizacion = DateTime.Now - tInicio;
            this.tiempoProceso += tiempoActualizacion;

            this.toolStripLabel1.Text = this.tiempoProceso.ToString();



            this.toolStripProgressBar1.Visible = false;

            this.listaGen.Clear();
            this.listaPK.Clear();
            this.listaIndices.Clear();
        }

        /// <summary>
        /// Crea las conexiones a las bases de datos
        /// </summary>
        /// <returns><b>true</b> si conecta con las dos correctamente, o <b>false</b> en caso de error</returns>
        private bool CreaConexiones()
        {
            try
            {
                FbConnectionStringBuilder csFb = new FbConnectionStringBuilder();

                csFb.DataSource = this.tbFbHost.Text;
                csFb.Database = this.tbFbDatabase.Text;
                csFb.UserID = this.tbFbUser.Text;
                csFb.Password = this.tbFbPassword.Text;
                this.connFirebird = new FbConnection(csFb.ToString());
                this.connFirebird.Open();
            }
            catch (FbException ex)
            {
                MessageBox.Show(ex.Message, "Error conexión Firebird");
                return false;
            }

            try
            {
                NpgsqlConnectionStringBuilder csPg = new NpgsqlConnectionStringBuilder();
                csPg.Host = this.tbPgHost.Text;
                csPg.Database = this.tbPgDatabase.Text;
                csPg.UserName = this.tbPgUser.Text;
                csPg.Password = this.tbPgPassword.Text;
                csPg.Port = 5432;
                csPg.Protocol = ProtocolVersion.Version3;
                csPg.SslMode = SslMode.Prefer;
                csPg.SSL = true;
                this.connPostgres = new NpgsqlConnection(csPg.ToString());
                this.connPostgres.Open();
            }
            catch (NpgsqlException ex)
            {
                MessageBox.Show(ex.Message, "Error conexión PostgreSQL");
                this.connFirebird.Close();
                return false;
            }


            return true;
        }

        /// <summary>
        /// Crea la tabla en el servidor PostgreSQL
        /// </summary>
        /// <param name="nombreTabla">Nombre de la tabla que vamos a crear</param>
        /// <param name="listaCampos">Lista de los campos de la tabla según los datos extraidos a Firebird</param>
        /// <returns><b>true</b> si crea la tabla correctamente, o <b>false</b> en caso contrario</returns>
        private bool CreaTabla(string nombreTabla, List<InfoEstructuraFirebird> listaCampos)
        {
            try
            {
                using (NpgsqlCommand cmd = this.connPostgres.CreateCommand())
                {
                    cmd.CommandText = String.Format("create table {0}({1}) without oids", nombreTabla, CreaCamposPostgreSQL(nombreTabla, listaCampos));
                    cmd.ExecuteNonQuery();

                    listaCampos.ForEach(
                        delegate(InfoEstructuraFirebird fb)
                        {
                            if (!String.IsNullOrEmpty(fb.Descripcion))
                            {
                                cmd.CommandText = String.Format("COMMENT ON COLUMN {0}.{1} IS '{2}'", nombreTabla, fb.Campo, fb.Descripcion);
                                cmd.ExecuteNonQuery();
                            }
                        });
                }
            }
            catch (NpgsqlException ex)
            {
                MessageBox.Show(ex.Message, String.Format("Error creando tabla {0}", nombreTabla));
                return false;
            }

            return true;
        }

        /// <summary>
        /// Trasdpasa los datos de la tabla indicada por el primer parámetro
        /// </summary>
        /// <param name="nombreTabla">Nombre de la tabla de la que vamos a traspasar los datos</param>
        /// <param name="lista">Lista de los campos de la tabla según los datos extraidos a Firebird</param>
        private void TraspasaDatosTabla(string nombreTabla, List<InfoEstructuraFirebird> lista)
        {
            bool quitaCamposNull = this.checkCamposNULL.Checked;
            string nombresCampos = InfoEstructuraFirebird.ListaCampos(lista, String.Empty);
            string nombresParametros = InfoEstructuraFirebird.ListaCampos(lista, ":");

            using (FbCommand cmdFirebird = this.connFirebird.CreateCommand())
            {
                using (NpgsqlCommand cmdPostgreSQL = this.connPostgres.CreateCommand())
                {
                    cmdFirebird.CommandText = String.Format("select count(*) from {0}", nombreTabla);
                    int numRegistros = (int)cmdFirebird.ExecuteScalar();

                    this.toolStripProgressBar1.Maximum = numRegistros;
                    this.toolStripProgressBar1.Value = 0;
                    this.toolStripProgressBar1.Step = 1;

                    cmdFirebird.CommandText = String.Format("select {0} from {1}", nombresCampos, nombreTabla);
                    cmdPostgreSQL.CommandText = String.Format("insert into {0} ({1}) values ({2})", nombreTabla, nombresCampos, nombresParametros);

                    lista.ForEach(
                        delegate(InfoEstructuraFirebird e)
                        {
                            if (e.Longitud != 0)
                                cmdPostgreSQL.Parameters.Add(new NpgsqlParameter(e.Campo, e.TipoDatosPostgresql(), e.Longitud));
                            else
                                cmdPostgreSQL.Parameters.Add(new NpgsqlParameter(e.Campo, e.TipoDatosPostgresql()));
                        });

                    //cmdPostgreSQL.Prepare();

                    using (FbDataReader dr = cmdFirebird.ExecuteReader())
                    {
                        while (dr.Read())
                        {
                            int numColumnas = dr.FieldCount;
                            for (int i = 0; i < numColumnas; i++)
                            {
                                if (lista[i].TipoDB.Equals("blob") && !lista[i].TipoDB.Equals("blob sub_type 1"))
                                {
                                    int noid = 0;

                                    if (!dr.IsDBNull(i))
                                    {
                                        NpgsqlTransaction t = this.connPostgres.BeginTransaction();
                                        LargeObjectManager lbm = new LargeObjectManager(this.connPostgres);

                                        noid = lbm.Create(LargeObjectManager.READWRITE);
                                        LargeObject lo = lbm.Open(noid, LargeObjectManager.READWRITE);

                                        lo.Write((byte[])dr.GetValue(i));
                                        lo.Close();
                                        t.Commit();
                                    }
                                    cmdPostgreSQL.Parameters[i].Value = noid;
                                }
                                else
                                {
                                    object valor = dr.GetValue(i);

                                    if (dr.IsDBNull(i) && quitaCamposNull)
                                    {
                                        if (lista[i].EsTipoNumerico())
                                            valor = 0;
                                        else if (lista[i].EstipoCaracter())
                                            valor = String.Empty;
                                    }

                                    cmdPostgreSQL.Parameters[i].Value = valor;

                                }
                            }
                            cmdPostgreSQL.ExecuteNonQuery();
                            this.toolStripProgressBar1.PerformStep();
                            Application.DoEvents();
                        }
                        dr.Close();
                    }

                }
            }
            Application.DoEvents();
        }

        /// <summary>
        /// Crea los índices de la tabla que indica el parámetro
        /// </summary>
        /// <param name="nombreTabla">Nombre de la tabla a la que vamos a crear los índices</param>
        private void CreaIndices(string nombreTabla)
        {
            try
            {
                using (NpgsqlCommand cmd = this.connPostgres.CreateCommand())
                {
                    this.listaIndices.ForEach(
                        delegate(TablasIndices indice)
                        {
                            if (nombreTabla.Equals(indice.Tabla))
                            {
                                cmd.CommandText = String.Format("CREATE {0} INDEX {1} ON {2}({3})",
                                                                indice.Unico ? "UNIQUE" : String.Empty,
                                                                indice.NombreIndice,
                                                                indice.Tabla,
                                                                indice.Campos);
                                cmd.ExecuteNonQuery();
                            }
                        });
                }
            }
            catch (NpgsqlException ex)
            {
                MessageBox.Show(ex.Message, String.Format("Error creando indice en {0}", nombreTabla));
            }
        }

        /// <summary>
        /// Actualiza los valores de las secuencias asociadas a los campos de tipo 'serial'
        /// </summary>
        private void ActualizaValoresSecuencias()
        {
            using (NpgsqlCommand cmd = this.connPostgres.CreateCommand())
            {
                //string sentencia1 = "SELECT seq.relname::text " +
                //                    "FROM pg_class src, pg_class seq, pg_namespace, pg_attribute,pg_depend " +
                //                    "WHERE " +
                //                    "pg_depend.refobjsubid = pg_attribute.attnum AND " +
                //                    "pg_depend.refobjid = src.oid AND " +
                //                    "seq.oid = pg_depend.objid AND " +
                //                    "seq.relkind = 'S' AND " +
                //                    "src.relnamespace = pg_namespace.oid AND " +
                //                    "pg_attribute.attrelid = src.oid AND " +
                //                    "src.relname = :tabla AND " +
                //                    "pg_attribute.attname = :campo";

                //cmd.Parameters.Add(new NpgsqlParameter("tabla", NpgsqlTypes.NpgsqlDbType.Text));
                //cmd.Parameters.Add(new NpgsqlParameter("campo", NpgsqlTypes.NpgsqlDbType.Text));

                this.toolStripProgressBar1.Maximum = this.listaGen.Count;
                this.toolStripProgressBar1.Value = 0;

                this.listaGen.ForEach(
                    delegate(TablasGeneradores g)
                    {
                        this.toolStripProgressBar1.PerformStep();

                        //cmd.Parameters["tabla"].Value = g.Tabla;
                        //cmd.Parameters["campo"].Value = g.Campo;
                        //cmd.CommandText = sentencia1;
                        cmd.CommandText = String.Format("SELECT pg_get_serial_sequence('{0}','{1}')", g.Tabla, g.Campo);
                        object o = cmd.ExecuteScalar();
                        if (o != DBNull.Value && o != null)
                        {
                            string nombre = o.ToString();
                            cmd.CommandText = String.Format("SELECT setval('{0}', {1}) FROM {2}",
                                                            nombre,
                                                            g.ValorActual,
                                                            g.Tabla);

                            cmd.ExecuteNonQuery();
                        }
                    });
            }
        }

        /// <summary>
        /// Devuelve una cadena de texto que contiene los campos y sus tipos adaptados a PostgreSQL.
        /// Utilizamos esta cadena de texto para crear la tabla en el servidor PostgreSQL.
        /// </summary>
        /// <param name="nombreTabla">Nombre de la tabla que vamos a crear</param>
        /// <param name="listaCampos">Lista de los campos de la tabla según los datos extraidos a Firebird</param>
        /// <returns>Cadena de texto que contiene los nombres de los campos y sus tipos, separados por comas</returns>
        private string CreaCamposPostgreSQL(string nombreTabla, List<InfoEstructuraFirebird> listaCampos)
        {
            bool primero = true;

            bool quitaCamposNull = this.checkCamposNULL.Checked;

            TablasGeneradores gen = this.listaGen.Find(
                delegate(TablasGeneradores g)
                {
                    return g.Tabla.Equals(nombreTabla);
                });

            bool campoSerialOk = false;

            if (gen == null)
                campoSerialOk = true;

            StringBuilder sb = new StringBuilder();
            listaCampos.ForEach(
                delegate(InfoEstructuraFirebird c)
                {
                    if (!campoSerialOk && gen.Campo.Equals(c.Campo))
                    {
                        sb.AppendFormat("{0}{1} serial", primero ? "" : ",", c.Campo);
                        campoSerialOk = true;
                    }
                    else
                    {
                        sb.AppendFormat("{0}{1} {2}", primero ? "" : ",", c.Campo, c.TipoPG);

                        if (c.Longitud != 0)
                        {
                            if (c.Precision == 0)
                                sb.AppendFormat("({0})", c.Longitud);
                            else
                                sb.AppendFormat("({0},{1})", c.Longitud, c.Precision);
                        }

                        if (!c.PermiteNull || (quitaCamposNull && (c.EstipoCaracter() || c.EsTipoNumerico())))
                            sb.Append(" not null");

                        if (!String.IsNullOrEmpty(c.ValorInicial))
                        {
                            sb.AppendFormat(" default {0}", c.ValorInicial);
                        }
                        else if (quitaCamposNull && (c.EstipoCaracter() || c.EsTipoNumerico()))
                        {
                            if (c.EstipoCaracter())
                                sb.Append(" default ''");
                            else if (c.EsTipoNumerico())
                                sb.Append(" default 0");
                        }
                    }
                    if (primero)
                        primero = !primero;
                });

            string camposPK = TablasClavesPrimarias.BuscaCamposTabla(this.listaPK, nombreTabla);
            if (!String.IsNullOrEmpty(camposPK))
            {
                sb.AppendFormat(",primary key({0})", camposPK);
            }

            return sb.ToString();
        }

        #endregion

        #region · DataGridViewEdit ·

        /// <summary>
        /// Rellena el DataGridView con los nombres de las tablas que vamos a traspasar
        /// </summary>
        /// <param name="dt">Objeto <see cref="DataTable"/> que contiene las tablas devueltas por Firebird</param>
        private void RellenaDataGridView(DataTable dt)
        {
            this.dataGridView1.Rows.Clear();

            int numFilas = dt.Rows.Count;
            for (int i = 0; i < numFilas; i++)
            {
                DataRow r = dt.Rows[i];
                if (Convert.ToInt32(r["IS_SYSTEM_TABLE"].ToString()) == 0)
                {
                    this.dataGridView1.Rows.Add();
                    this.dataGridView1.Rows[i].Cells[(int)ColumnasDgv.Nombre].Value = r["TABLE_NAME"].ToString().ToLowerInvariant();
                }
            }

            this.dataGridView1.Refresh();

            this.numTablasDgv = this.dataGridView1.RowCount;
        }

        /// <summary>
        /// Activa la marca de la celda actual, y cambia ésta a la siguiente celda.
        /// Si estamos en la columna de índices, cambia a la siguiente línea
        /// </summary>
        private void ColumnaOK()
        {
            if (this.dataGridView1.CurrentRow != null)
            {
                int iRow = this.dataGridView1.CurrentRow.Index;
                int iCol = this.dataGridView1.CurrentCell.ColumnIndex;

                this.dataGridView1[iCol, iRow].Value = true;

                if (++iCol > (int)ColumnasDgv.Indice)
                {
                    iCol = (int)ColumnasDgv.Tabla;
                    iRow++;
                    if (iRow >= this.numTablasDgv)
                        return;
                }

                this.dataGridView1.CurrentCell = this.dataGridView1[iCol, iRow];

                this.dataGridView1.Refresh();
            }
        }

        #endregion
    }
}
