#include <arpa/inet.h>
#include <net/ethernet.h>
#include <netdb.h>
#include <netinet/ether.h>
#include <netinet/if_ether.h>
#include <netinet/ip.h>
#include <pthread.h>
#include <sys/time.h>
#include <unistd.h>
#include <signal.h>
#include <syslog.h>
#include <fcntl.h>


#include <iostream>
#include <map>
#include <list>
#include <string>

extern "C" {
#include <pcap.h>
}

using namespace std;

#include "sqlwrite.h"
#include "ip.h"
#include "sll.h"
#include "dns.h"
#include "prio.h"

static bool	FlushDataToDb(RTable *rt, int since)
{
	struct timeval tv;
	bool seen = false;
	CRecordList	crl;

	gettimeofday(&tv, NULL);
	tv.tv_sec -= since;
	rt->MakeListOfFlushItems(&crl, &tv);
	for (CRecordListIter iter = crl.begin(); iter != crl.end(); ) {
		CRecordListIter prev = iter;
		CRecord *cr = *iter++;
		cr->srcSerial_m = rt->FindHostFromIP(cr->ip_src_m, cr->llSrc_m);
		cr->dstSerial_m = rt->FindHostFromIP(cr->ip_dst_m, cr->llDst_m);
		if (SQlSaveRecord(cr, rt->Get_InterfaceName()) != 0) {
			seen = true;
			crl.erase(prev);
			delete cr;
		} 
	}
	return seen;
}

static void	SetSignals(bool allowControlC)
{
	sigset_t  blockedSigs;

	/* Get the full signal set */
	sigfillset(&blockedSigs);

	if (allowControlC)
		sigdelset(&blockedSigs, SIGINT);
	/* Block all signals (except, possibly, 34) */
  pthread_sigmask(SIG_BLOCK, &blockedSigs, NULL);
}

static void    *SaveDataThread(void *arg)
{
	RTable	*rt = (RTable *)arg;

	SetSignals(false);
	for (;;) {
		for (int ii = 0; ii < rt->Get_SleepTime(); ii++) {
			sleep(1);
			pthread_testcancel();
		}
		FlushDataToDb(rt, rt->Get_Age());
		rt->Get_DnsCache()->FlushOld();
	}
}


u_char *handle_IP(u_char *args, const struct pcap_pkthdr *pkthdr, const u_char * packet, int hrdSize,
	const char *llSrc, const char *llDst)
{
  const struct my_ip *ip;
  u_int length = pkthdr->len;
  u_int hlen = 0; 
  u_int	off = 0;
  u_int	version = 0;
//  int i;

  int len;

/* jump pass the ethernet header */
  ip = (struct my_ip *)(packet + hrdSize);
  length -= hrdSize;

/* check to see we have a packet of valid length */
	if (length < sizeof (struct my_ip)) {
		syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_INFO), "truncated ip of length %d", length);
		return NULL;

	}
	len = ntohs(ip->ip_len);
	hlen = IP_HL(ip);		/* header length */
	version = IP_V(ip);		/* ip version */

	/* check version */
	if (version != 4) {
		syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_INFO), "unsupported IP version %d", version);
		return NULL;
	}

	/* check header length */
	if (hlen < 5) {
		syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_INFO), "bad-hlen  %d", hlen);
		return NULL;
	}

	/* see if we have as much packet as we should */
	if (length < len) {
		syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_INFO), "truncated IP - %d bytes missing", len - length);
		return NULL;
	}

	RTable  *rt = (RTable *)args;

	CRecord	*cr = new CRecord(ip, pkthdr->len, llSrc, llDst);
	rt->Add(cr, pkthdr->ts);
	return NULL;
}

#ifndef ETHER_HDRLEN
#define ETHER_HDRLEN 14
#endif

void my_callback(u_char *args, const struct pcap_pkthdr *pkthdr, const u_char *packet)
{
	RTable *rt = (RTable *)args;
	u_int caplen = pkthdr->caplen;
	u_int length = pkthdr->len;

	switch (rt->Get_DataLinkType()) {
	case DLT_PPP:
		break;

	case DLT_LINUX_SLL: {
			const struct sll_header *sllp = (const struct sll_header *)packet;
			u_short ether_type = ntohs(sllp->sll_protocol);
			switch (ether_type) {
			case ETHERTYPE_IP: {
					handle_IP(args, pkthdr, packet, SLL_HDR_LEN, NULL, NULL);
					break;
				}
			default:
				syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_INFO), "unhandled ethertype: 0");
				break;
			}
		}
		break;


	case DLT_EN10MB: {
			const struct ether_header *eptr = (const struct ether_header *)packet;
			u_int16_t type = ntohs(eptr->ether_type);
			char	llSrc[80];
			char	llDst[80];
			if (caplen < ETHER_HDRLEN) {
				syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_INFO), "Packet length less than ethernet header length");
				return;
			}
			switch (type) {
			case 0:
				syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_INFO), "unhandled ethertype: 0");
				break;
			case ETHERTYPE_IP:
				ether_ntoa_r((struct ether_addr *)eptr->ether_shost, llSrc);
				ether_ntoa_r((struct ether_addr *)eptr->ether_dhost, llDst);
				handle_IP(args, pkthdr, packet, sizeof (struct ether_header), llSrc, llDst); //llSrc, llDst);
				break;
			case ETHERTYPE_ARP:
				break;
			case ETHERTYPE_REVARP:
				break;
			default:
				break;
			}
		}
		break;
	default:
		syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_INFO), "unknown data link layer type %d",  rt->Get_DataLinkType());
		break;
	}
}

RTable   *rt;
pthread_t	writeThread;
 

static void	FinalFlush()
{
	syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_INFO), "got signal waiting for child to die off");
	pthread_cancel(writeThread);
	
	syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_INFO), "flushing");
	if (FlushDataToDb(rt, 0)) {
		syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_INFO), "committed further records");
	} else {
		syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_INFO), "not committed further records");
	}
}

static void termination_handler(int signum)
{
	FinalFlush();

	struct pcap_stat ps;
	if (rt->Get_descr() != 0) {
		if (pcap_stats(rt->Get_descr(), &ps) < 0) {
			cerr << "pcap_stats failed: " << pcap_geterr(rt->Get_descr()) << endl;
			exit(1);
		}
	}
	FinalFlush();
	syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_INFO), "%d recived and %d dropped", ps.ps_recv, ps.ps_drop);
	exit(0);
}

int main(int argc, char *argv[])
{
    int nerr;

	char *dev = (char *)0;
	char errbuf[PCAP_ERRBUF_SIZE];

	int	opt;
	int	sltime = 10;
	int	numPackets = 10;
	int	age = 30;
	char	*connectionString = "usage@localhost";
	char	*filter = NULL;
	int		dnsExpiryTime = 30;
	int	backGround = false;
	int	ifDownSleep = 10;

	while ((opt = getopt(argc, argv, "S:bx:s:p:c:f:d:i:a:")) != -1) {
		switch (opt) {
		case 'S':
			ifDownSleep = atoi(optarg);
			break;
		case 'b':
			backGround = true;
			break;
		case 'x':
			dnsExpiryTime = atoi(optarg);
			break;
		case 's':
			sltime = atoi(optarg);
			break;
		case 'p':
			numPackets = atoi(optarg);
			break;
		case 'c':
			connectionString = optarg;
			break;
		case 'f':
			filter = optarg;
			break;
		case 'd':
		case 'i':
			dev = optarg;
			break;
		case 'a':
			age = atoi(optarg);
			break;
		default:
			cerr << "invalid usage" << endl;
			cerr << "-b - fork in background and detatch from console" << endl;
			cerr << "-x dns expiry time: the time an entry lives int he dns cache" << endl;
			cerr << "-s sleep_time: the time the write sleeps for" << endl;
			cerr << "-p num_packets: the number of packets to collect, 0 for forever" << endl;
			cerr << "-f filter: pcap filter argument" << endl;
			cerr << "-d or -i: device (or interface) to capure on" << endl;
			cerr << "-a: age of to/from packets inactivity before writing to db" << endl;
			cerr << "-c: connection string" << endl;
			cerr << "-S: number of seconds to sleep if interface is down" << endl;
			exit(1);
		}
	}
	SQLConnect(connectionString);
	if (dev == (char *)0) {
		if ((dev = pcap_lookupdev(errbuf)) == (char *)0) {
			cerr << "pcap_lookupdev failed: " << errbuf << endl;
			exit(1);
		}
	}
	if (backGround) {
		int fval =fork();
		if (fval < 0) {
			perror("fork");
			exit(1); /* fork error */
		} else if (fval > 0) {
			syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_INFO), "forked child %d", fval);
			exit(0); /* parent exits */
		}
		setsid();
	}

	syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_INFO), "capturing on interface '%s'", dev);

	bpf_u_int32 maskp;    /* subnet mask   */
	bpf_u_int32 netp;     /* ip   */           
	pcap_t *descr;			/* pcap descriptor */

	if (pcap_lookupnet(dev, &netp, &maskp, errbuf) < 0) {
		cerr << "pcap_lookupnet(): " << errbuf << endl;
		exit(1);
	}
	struct bpf_program compiledFilter;

	if (filter != 0) {
		if (pcap_compile(descr, &compiledFilter, filter, 0, netp) == -1) {
			cerr << "Error calling pcap_compile" << endl;
			exit(1);
		}
	}

	DnsCache	dnsCache(dnsExpiryTime);
	rt = new RTable(sltime, &dnsCache, age, dev);

   if (pthread_create(&writeThread, NULL, SaveDataThread, (void *)rt) < 0) {
		perror("pthread_create");
		exit(1);
    }
	signal(SIGINT, termination_handler);
	signal(SIGTERM, termination_handler);

	for (;;) {
		if ((descr = pcap_open_live(dev, BUFSIZ, 1, -1, errbuf)) == (pcap_t *)0) {
			syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_INFO), "pcap_open_live(): %s", errbuf);
			if (numPackets != 0)
				exit(1);
		} else {
			syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_INFO), "capturing datalink type %d", pcap_datalink(descr));
			rt->Set_tcapInfo(descr);
			if (filter != 0) {
				/* set the compiled program as the filter */
				if (pcap_setfilter(descr, &compiledFilter) == -1) {
					cerr << "Error setting filter" << endl;
					exit(1); 
				} 
			}

			pcap_loop(descr, numPackets, my_callback, (u_char *)rt);
			if (numPackets != 0)
				break;
		}
		syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_INFO), "down .. sleeping %d", ifDownSleep);
		sleep(ifDownSleep);
	}
	FinalFlush();

	return 0;
}


