package pgclibrary.dataaccess;

import java.sql.*;
import java.util.*;
import java.io.*;
import java.text.*;

/**
 * Clase que inicializa la conexion del usuario a la base de datos.
 * Realiza una abtracci\u00f3n de los m\u00e9todos mas utilizados en una coneccion.
 *
 * <H4> Este objeto no debe manejar procedimientos Batch. </H4>
 *
 * @author Nestor Marsollier
 * @version 1.1
 */
public class MainConnection extends Object implements Serializable {
    public static final String defaultDriver = "org.postgresql.Driver";
    public static final String defaultURI = "jdbc:postgresql://localhost:5432/prueba1";
    
    private String driver = null;                  // Driver utilizado en la conexion
    private String URI = null;                      // Origen de datos utilizado
    private String userName = null;            // Nombre del usuario conectado
    private String password = null;             // Contrase\u00f1a utilizada en la conexion
    private Connection connection = null;   // java.sql.Connection utilizado
    private Statement statement = null;
    
    private Vector error_text = new Vector();
    private long error_code = 0;
    
    /*
     * Constructor sin par\u00e1metros
     */
    public MainConnection(){}
    /**
     * Constructor completo.
     * Inicializa la conexion a una base de datos determinada por los par\u00e1metros.
     *
     * @param _driver  - Driver de base de datos utilizado
     * @param _URI  - Direccion URL de la base de datos que se intenta conectar
     * @param _userName  - Nombre del usuario de la base de datos
     * @param _password  - Contrase\u00f1a utilizada en la conexi\u00f3n
     */
    public MainConnection(String _driver, String _URI, String _userName, String _password){
        error_text = new Vector();
        error_code = 0;
        
        if((_driver != null) && (_URI != null) && (_userName != null)
        && (_password != null)){
            driver = new String(_driver);
            URI = new String(_URI);
            userName = new String(_userName);
            password = new String(_password);
            
            Properties props = new Properties();
            props.put("user",userName);
            props.put("password",password);
            
            connect(props);
        }
    }
    
    
    public boolean changePassword(String _newPassword){
        boolean retorno = false;
        error_text = new Vector();
        error_code = 0;
        
        if((_newPassword != null) && (_newPassword.length() > 0)){
            Properties props = new Properties();
            props.put("user",userName);
            props.put("password",password);
            props.put("OCINewPassword", _newPassword);
            retorno = connect(props);
            if(retorno){
                password = _newPassword;
            }
        } else {
            error_text.add("La contrase&ntilde;a proporcionada es incorrecta, o se encuentra deshabilitada.");
        }
        return retorno;
    }
    
    /* Realiza la conexion y maneja los errores que pueden surgir
     **/
    private boolean connect(Properties props){
        boolean retorno = false;
        
        Connection temp = null;
        
        if((driver != null) && (URI != null) && (userName != null) && (password != null)){
            try{
                Class.forName(driver);
                temp = DriverManager.getConnection(URI, props);
                temp.setAutoCommit(true);
                SQLWarning sqlw = temp.getWarnings();
                if(sqlw != null && sqlw.getErrorCode() == 28002){
                    error_text.add("La contrase&ntilde;a est&aacute; a punto de expirar, por favor c&aacute;mbiela.");
                    error_code = sqlw.getErrorCode();
                }
            }catch(Exception e){
                error_text.add(e.getMessage());
                temp = null;
            }
        }
        if(temp != null){
            connection = temp;
            try{
                statement = connection.createStatement(ResultSet.TYPE_SCROLL_INSENSITIVE, ResultSet.CONCUR_UPDATABLE);
                retorno = true;
            } catch (Exception e){
                error_text.add(e.getMessage());
                connection = null;
                retorno = false;
            }
        }
        return retorno;
    }
    
    
    /**
     * Constructor a la base de datos predeterminada.
     * Inicializa la conexion a la base de datos determinada las
     * definiciones defaultDriver y defaultURI
     *
     * @param _userName  - Nombre del usuario de la base de datos
     * @param _password  - Contrase\u00f1a utilizada en la conexi\u00f3n
     */
    public MainConnection(String _userName, String _password){
        this(defaultDriver, defaultURI, _userName, _password);
    }
    
    /**
     * Constructor a partir de otro objeto MainConnection.
     * @param _mainConnection Objeto a clonar
     */
    public MainConnection(MainConnection _mainConnection){
        this(_mainConnection.getDriver(),
        _mainConnection.getURI(),
        _mainConnection.getUserName(),
        _mainConnection.getPassword());
    }
    
    public Vector getError(){
        return error_text;
    }
    public long getErrorCode(){
        return error_code;
    }
    
    /**
     * Devuelve el Nombre de Usuario utilizado en la conexi\u00f3n.
     *
     * @return Nombre del usuario conectado
     */
    public String getUserName(){
        return userName;
    }
    
    /**
     * Devuelve el Nombre de Usuario utilizado en la conexi\u00f3n.
     *
     * @return Nombre del usuario conectado
     */
    private String getPassword(){
        return password;
    }
    
    /**
     * Devuelve el objeto connection que maneja la conexion
     *
     * @return Objeto java.sql.Connection.
     * @throws SQLException Si no est\u00e1 conectado.
     */
    public Connection getConnection() throws SQLException{
        if (!isConnected()){
            throw new SQLException("La conexi\u00f3n no ha sido inicializada");
        }
        return connection;
    }
    
    /**
     * Recupera el Driver que se est\u00e1 utilizando en la conexion.
     *
     * @return String Corresponde al Driver de conexi\u00f3n
     */
    public String getDriver(){
        return driver;
    }
    
    /**
     * Recupera origen de datos de la conexi\u00f3n.
     *
     * @return String Origen de informaci\u00f3n de la conexi\u00f3n actual.
     */
    public String getURI(){
        return URI;
    }
    
    /**
     * Identifica el estado de la conexi\u00f3n.
     *
     * @return true Si est\u00e1 conectado a la base de datos
     * @return false Si no se pudo conectar
     */
    public boolean isConnected(){
        return connection != null;
    }
    
    /**
     * Crea y adevualve un Statement de conexion ReadOnly.
     * Crea y devuelve un nuevo Statement de conexi\u00f3n
     * con atributos ReadOnly. Puede ser \u00fatil para consultar.
     *
     * @return Statement
     * @throws SQLException Si no est\u00e1 conectado
     */
    public Statement getStatementReadOnly() throws SQLException{
        Statement st = null;
        st = getConnection().createStatement(ResultSet.TYPE_SCROLL_INSENSITIVE,
        ResultSet.CONCUR_READ_ONLY);
        return st;
    }
    
    /**
     * Crea y devuelve un nuevo Statement de conexi\u00f3n.
     *
     * @return Statement
     * @throws SQLException Si no est\u00e1 conectado
     */
    public Statement getStatement() throws SQLException{
        Statement st = null;
        st = getConnection().createStatement(ResultSet.TYPE_SCROLL_INSENSITIVE, ResultSet.CONCUR_READ_ONLY);
        return st;
    }
    
    /**
     * Crea y devuelve un nuevo Statement de conexi\u00f3n navegable.
     *
     * @return Statement
     * @param tipo Tipo de ResultSet
     * @param concur Tipo de concurrencia
     * @throws SQLException Si no est\u00e1 conectado
     */
    public Statement getStatement(int tipo, int concur) throws SQLException{
        Statement st = null;
        st = getConnection().createStatement(tipo, concur);
        return st;
    }
    
    /**
     * Ejecuta una sentencia en la Base de datos y devuelve
     * la cantidad de registros modificados.
     * Corresponde a una abstraccion del m\u00e9todo
     *
     * Es una abtracci\u00f3n de Statement.executeUpdate()
     *
     * @return int cantidad de registros modificados
     * @param sql Consulta SQL a ejecutar
     * @throws SQLException Generado por el Statement
     */
    public int executeUpdate(String sql) throws SQLException{
        if(statement != null){
            return statement.executeUpdate(sql);
        } else {
            return 0;
        }
    }
    
    /**
     * Ejecuta una consulta de la base de datos y devuelve el resultado
     * en un resultset
     *
     * Es una abtracci\u00f3n de Statement.executeQuery()
     *
     * @return ResultSet
     * @param sql Consulta a Ejecutar
     * @throws SQLException Genrado por Statement
     */
    public ResultSet executeQuery(String sql) throws SQLException {
        if(statement != null){
            return statement.executeQuery(sql);
        } else {
            return null;
        }
    }
    
    /**
     * Prepara un Statement.
     *
     * Es una abstraccion de Connection.prepareStatement()
     *
     * @return PreparedStatement.
     * @param sql Consulta a Ejecutar
     * @throws SQLException Generado por Connection
     */
    public PreparedStatement prepareStatement(String sql) throws SQLException{
        if (!isConnected()){
            throw new SQLException("La conexi\u00f3n no ha sido inicializada");
        }
        return getConnection().prepareStatement(sql, ResultSet.TYPE_SCROLL_INSENSITIVE, ResultSet.CONCUR_READ_ONLY);
    }
    
    /**
     * Prepara un Statement navegable.
     *
     * Es una abstraccion de Connection.prepareStatement()
     *
     * @return PreparedStatement.
     * @param sql Consulta a Ejecutar
     * @param tipo Tipo de ResultSet
     * @param concur Tipo de concurrencia
     * @throws SQLException Generado por Connection
     */
    public PreparedStatement prepareStatement(String sql, int tipo, int concur) throws SQLException{
        if (!isConnected()){
            throw new SQLException("La conexi\u00f3n no ha sido inicializada");
        }
        return getConnection().prepareStatement(sql, tipo, concur);
    }
    
    /**
     * Cierra correctamente la conexi\u00f3n
     * debe ser llamado antes de eliminar un objeto MainConnection
     */
    public void close(){
        if (isConnected()){
            try{
                statement.close();
                connection.close();
            } catch (SQLException e){};
        }
    }
    
    protected void finalize() throws Throwable{
        close();
    }
    
    public static boolean isQuery(String _query){
        String tmp = _query.toUpperCase();
        return (tmp.indexOf("SELECT") >= 0) 
                    && (tmp.indexOf("UPDATE") < 0)
                    && (tmp.indexOf("DELETE") < 0)
                    && (tmp.indexOf("GRANT") < 0)
                    && (tmp.indexOf("REVOKE") < 0)
                    && (tmp.indexOf("INSERT") < 0);
    }
    
    public String getDatabaseName(){
        String retorno = new String();
        if(isConnected()){
            StringTokenizer stok = new StringTokenizer(getURI(), "/");
            
            while(stok.hasMoreTokens()){
                retorno = stok.nextToken();
            }
        }
        return retorno;
    }
    public String getProductName(){
        String retorno = new String();
        try{
            if(isConnected()){
                retorno = getConnection().getMetaData().getDatabaseProductName() + " " +
                getConnection().getMetaData().getDatabaseProductVersion();
            }
        } catch (SQLException e){}
        return retorno;
    }
    
    public String getDriverVersion(){
        String retorno = new String();
        try{
            if(isConnected()){
                retorno = getConnection().getMetaData().getDriverVersion();
            }
        } catch (SQLException e){}
        return retorno;
    }
    
    public String getURL(){
        String retorno = new String();
        try{
            if(isConnected()){
                retorno = getConnection().getMetaData().getURL();
            }
        } catch (SQLException e){}
        return retorno;
    }
    
    static {
        Locale.setDefault(Locale.US);
    }
}


